// CRTSIS.H
// Copyright (c) 1997-1999 Symbian Ltd. All rights reserved.
//
// Handles creation of a SIS file
//

#ifndef __CRTSIS_H_
#define __CRTSIS_H_

// ===========================================================================
// INCLUDES
// ===========================================================================

#include <instform.h>

#include "parsepkg.h"
#include "zlib.h"


// ===========================================================================
// DEFINES
// ===========================================================================

// TGeneratorError
// What could possibly go wrong ?
enum TGeneratorException
	{
	ErrFailedToWriteHeader,			
	ErrFailedToWriteLanguages,		
	ErrFailedToWriteFilesBlock,		
	ErrFailedToWriteDependencyBlock,
	ErrFailedToWriteCapabilitiesBlock,
	ErrFailedToWriteOptionsBlock,
	ErrFailedToWriteConditionBlock,
	ErrFailedToWriteCertificatesBlock,
	ErrFailedToWriteSignatureBlock,
	ErrFailedToWriteStringsBlock,	
	ErrFailedToWriteCRC,
	ErrFailedToOpenStubTarget,
	ErrFailedCompression,
	};

// Define return conditions from PKCS7 class
#define	PKCS7FileIsCertificate 1
#define PKCS7NotPCKS7SignedFile 2
#define PKCS7FileReWritten 255




// ===========================================================================
// CLASS DEFINITIONS
// ===========================================================================

class CPrivateKey;


class CSISWriter
// Holds SIS file info, as generated by the parser and builds the output SIS file.
// All strings are held internally by this class as UNICODE (wide) strings.
	{
	public:
		CSISWriter();
		void Release();											// Destroy contained objects and free memory

		void WriteSIS(LPCWSTR pszTargetFile, BOOL fIsStub = FALSE);		// Construct the output SIS file
		
		void AddLanguageNode(LANGNODE *pLNode);
		void AddPkgLineNode(PKGLINENODE *pPkgNode);
		void AddDependencyNode(DEPENDNODE *pNode);
		void AddSignatureNode(SIGNATURENODE *pSig);
		void AddCapabilityNode(CAPABILITYNODE *pNode);
		void AddLangStringNode(LANGSTRINGNODE *pLSNode);

		void SetCmdPassword(LPCWSTR pszCmdPassword);
		void SetObserver(const MParserObserver *pObserver) { m_pObserver = pObserver; }
		void SetVersionInfo(DWORD dwUID, WORD wMajor, WORD wMinor, DWORD dwBuild, TSISType wType, WORD wFlags);
		void SetLangDependFileDestinations(LPCWSTR pszDest, LPCWSTR pszMimeType, DWORD dwType, DWORD dwOptions);
		void SetLangDependCompDestinations(DWORD dwUid);

		BOOL GetCmdPassword(LPWSTR pszCmdPassword) const;
		const MParserObserver* GetObserver()		const { return m_pObserver; };		
		const LANGSTRINGNODE* GetLangStringBase()	const { return m_pLangStringBase; }
		const LANGNODE* GetLanguageBase()			const { return m_pLangBase; }
		const PKGLINENODE* GetPkgLineBase()			const { return m_pPkgLineBase; }
		const DEPENDNODE* GetDependencyBase()		const { return m_pDependBase; }
		const SIGNATURENODE* GetSignatureBase()		const { return m_pSignatureBase;}
		const CAPABILITYNODE* GetCapabilityBase()	const { return m_pCapabilityBase; }
		DWORD GetUID()								const { return m_dwUID; }
		TSISType GetType()							const { return m_wType; }
		WORD GetFlags()								const { return m_wFlags; }
		WORD GetNoLanguages()						const { return m_wNumLangs; }

		void QueryVersionInfo (WORD* wMajor, WORD* wMinor, DWORD* dwBuild) const;
		BOOL AreLangStringInit();		
		
	private:
		void ReleaseCondExpr(PKGLINECONDITION* expr);				// Destroy condition expr
		
	private:
		// Installation data
		DWORD m_dwUID;						 // Installee app's UID
		WORD m_wMajor;						 // Version info
		WORD m_wMinor;						 //	   "	 "
		DWORD m_dwBuild;					 //	   "	 "
		TSISType m_wType;					 // SIS type (app, system, patch,...)
		WORD m_wFlags;						 // Installation flags (Unicode, redist ++)
		WORD m_wNumLangs;					 // How many languages do we speak ?
		
		LANGSTRINGNODE* m_pLangStringBase;	 // Language strings - the app names in N languages
		LANGNODE* m_pLangBase;				 // Language data - which langauges are supported
		PKGLINENODE* m_pPkgLineBase;		 // Linked list of PKG line nodes
		DEPENDNODE* m_pDependBase;			 // Dependency data
		SIGNATURENODE* m_pSignatureBase;	 // Digital signature data
		CAPABILITYNODE* m_pCapabilityBase;	 // Capability data
		BOOL m_bCmdPassword;				 // Commnand Line Password Flag
		WCHAR m_pszCmdPassword[MAX_PATH];	 // Command Line Decryption Password
		
		const MParserObserver *m_pObserver;	 // Observer object
	};

class CSISFileGeneratorBase
// Handles most of the SIS file generation operation. Defers the character width
// specific stuff to a derived template class...
	{
	public:
		BOOL m_bWriteChain;
		void ParseCertificates();
		CSISFileGeneratorBase(const CSISWriter *pSISWriter);
		virtual ~CSISFileGeneratorBase();
		
		void GenerateSISFile(LPCWSTR pszTargetFile, BOOL fIsStub);
		
	protected:
		void CheckUnicodeTextFiles();
		void CompressFiles();
		static void CompressFile(const PKGLINENODE *pNode, LPWSTR pszOutputFname, int &compressedSize);
		static void ReadFromUncompressedFile(HANDLE hFile,BYTE *pbBuffer,z_stream &compressionState, int bufferSize);
		static void WriteToCompressedFile(HANDLE hFile,BYTE *pbBuffer,z_stream &compressionState, int bufferSize);
		void CalculateOffsets();
		DWORD CalcConditionExprSize(const PKGLINECONDITION* expr, DWORD* stringSizes);

		void EstimateInstallationSizeL();
		int CalculateInstallSizeOfIfBlockL(const PKGLINENODE **pNode, int &index) const;
		int CalculateInstallSizeOfFileL(const PKGLINENODE *pNode) const; 
		void WriteHeaderL();
		void WriteCertificatesL();
		void WriteLanguageL();
		void WritePkgLinesL();
		void WriteFileNodeL(const PKGLINENODE* pNode);
		void WriteOptionsNodeL(const PKGLINENODE* pNode);
		void WriteConditionNodeL(const PKGLINENODE* pNode);
		void WriteCondExprL(const PKGLINECONDITION* expr);
		void WriteDependL();
		void WriteCapabilitiesL();
		void WriteStringsL();
		void WriteCondExprStrL(const PKGLINECONDITION* expr);
		void WriteCodeL();
		void WriteCrcL();
		void WriteSignatureL(LPCWSTR pszTargetFile);
		
		// Load the private key from the file
		CPrivateKey* LoadPrivateKey();

		void Crc(const void *pPtr, DWORD dwLength);	   // Keep a running CRC value
		void UIDCheck(const void *pPtr);				   // Calc the UID checksum value
		
		void ConfigureFileNode(const PKGLINEFILE* pFNode, TInstFile *pifFile);
		
		// Defer these next two functions to the derived template class
		virtual void WriteString(LPCWSTR pStr, HANDLE hFile) = 0;
		virtual void WriteString8(LPCWSTR pStr, HANDLE hFile) = 0;
		virtual int SizeOfChar() = 0;
		
	private:
		inline void Verbage(const _TCHAR* psz); // Emit verbose output	
		
	private:
		const CSISWriter* m_pSISWriter;	 // The parent CSISWriter object
		HANDLE m_hFile;					 // The output file handle

		BOOL m_stubFile;

		// Offset data, for the blocks
		DWORD m_dwOffLang;
		DWORD m_dwOffPkgLines;
		DWORD m_dwOffDepend;
		DWORD m_dwOffCertificates;
		DWORD m_dwOffCapabilities;
		DWORD m_dwOffString;
		DWORD m_dwOffCode;
		DWORD m_dwOffSignature;
		
		// Temporary offsets
		DWORD m_dwCurrString;	// The next string write pos (where we expect to write the next string)
		DWORD m_dwCurrCode;		// The next code block write pos (where we expect to store the next file)
		
		// CRC check
		WORD m_wCheck;			// The CRC checksum value

		DWORD m_dwMaximumSpaceNeeded;
	};

inline void CSISFileGeneratorBase::Verbage(const _TCHAR* psz) 
	{ 
	if(m_pSISWriter->GetObserver() != NULL)
		m_pSISWriter->GetObserver()->DoVerbage(psz); 
	}

template<class T>
class CSISFileGeneratorT : public CSISFileGeneratorBase
// Template class, handles the character size specific SIS file generation
	{
	public:
		CSISFileGeneratorT(const CSISWriter *pSISWriter) : CSISFileGeneratorBase(pSISWriter) {;}
		
	protected:
		virtual void WriteString(LPCWSTR pStr, HANDLE hFile);	// Put a string into the string block
		virtual void WriteString8(LPCWSTR pStr, HANDLE hFile);	// Put a string into the string block
		virtual int  SizeOfChar()  { return sizeof(T); };		// How big is a char ?
	};

#endif // __CRTSIS_H_
