// CRTSIS.CPP
// Copyright (c) 1997-1999 Symbian Ltd. All rights reserved.
//
// Handles creation of a SIS file
//

// ===========================================================================
// INCLUDES
// ===========================================================================

#include "utils.h"

#include <windows.h>
#include <assert.h>

#include <tchar.h>

#include <wchar.h>
#include <time.h>
#include <stdio.h>
#include "crtsis.h"
#include "parsepkg.h"
#include "instendian.h"

//#ifndef DIGITAL_SIGNATURES
//#define DIGITAL_SIGNATURES
//#endif


#ifdef DIGITAL_SIGNATURES
#undef DIGITAL_SIGNATURES
#endif


#ifdef DIGITAL_SIGNATURES
// implies you ''have'' to use wincrypt.lib 
#include <WinHash.h>
#include <WinKey.h>
#include <WinSHA1.h>
#include <WinDSA.h>
#include <WinEncrypt.h>
#include <winparsepkcs7.h>
#include <winB64Decrypt.h>
#endif

// ===========================================================================
// CONSTANTS
// ===========================================================================

#define HASHLEN 256
#define CIPHERTEXTLEN 2048

#define Y2K_FIX 1900




// CRC calulation lookup table
const WORD crcTab[256] =
	{0x0000,0x1021,0x2042,0x3063,0x4084,0x50a5,0x60c6,0x70e7,0x8108,0x9129,0xa14a,
	0xb16b,0xc18c,0xd1ad,0xe1ce,0xf1ef,0x1231,0x0210,0x3273,0x2252,0x52b5,0x4294,
	0x72f7,0x62d6,0x9339,0x8318,0xb37b,0xa35a,0xd3bd,0xc39c,0xf3ff,0xe3de,0x2462,
	0x3443,0x0420,0x1401,0x64e6,0x74c7,0x44a4,0x5485,0xa56a,0xb54b,0x8528,0x9509,
	0xe5ee,0xf5cf,0xc5ac,0xd58d,0x3653,0x2672,0x1611,0x0630,0x76d7,0x66f6,0x5695,
	0x46b4,0xb75b,0xa77a,0x9719,0x8738,0xf7df,0xe7fe,0xd79d,0xc7bc,0x48c4,0x58e5,
	0x6886,0x78a7,0x0840,0x1861,0x2802,0x3823,0xc9cc,0xd9ed,0xe98e,0xf9af,0x8948,
	0x9969,0xa90a,0xb92b,0x5af5,0x4ad4,0x7ab7,0x6a96,0x1a71,0x0a50,0x3a33,0x2a12,
	0xdbfd,0xcbdc,0xfbbf,0xeb9e,0x9b79,0x8b58,0xbb3b,0xab1a,0x6ca6,0x7c87,0x4ce4,
	0x5cc5,0x2c22,0x3c03,0x0c60,0x1c41,0xedae,0xfd8f,0xcdec,0xddcd,0xad2a,0xbd0b,
	0x8d68,0x9d49,0x7e97,0x6eb6,0x5ed5,0x4ef4,0x3e13,0x2e32,0x1e51,0x0e70,0xff9f,
	0xefbe,0xdfdd,0xcffc,0xbf1b,0xaf3a,0x9f59,0x8f78,0x9188,0x81a9,0xb1ca,0xa1eb,
	0xd10c,0xc12d,0xf14e,0xe16f,0x1080,0x00a1,0x30c2,0x20e3,0x5004,0x4025,0x7046,
	0x6067,0x83b9,0x9398,0xa3fb,0xb3da,0xc33d,0xd31c,0xe37f,0xf35e,0x02b1,0x1290,
	0x22f3,0x32d2,0x4235,0x5214,0x6277,0x7256,0xb5ea,0xa5cb,0x95a8,0x8589,0xf56e,
	0xe54f,0xd52c,0xc50d,0x34e2,0x24c3,0x14a0,0x0481,0x7466,0x6447,0x5424,0x4405,
	0xa7db,0xb7fa,0x8799,0x97b8,0xe75f,0xf77e,0xc71d,0xd73c,0x26d3,0x36f2,0x0691,
	0x16b0,0x6657,0x7676,0x4615,0x5634,0xd94c,0xc96d,0xf90e,0xe92f,0x99c8,0x89e9,
	0xb98a,0xa9ab,0x5844,0x4865,0x7806,0x6827,0x18c0,0x08e1,0x3882,0x28a3,0xcb7d,
	0xdb5c,0xeb3f,0xfb1e,0x8bf9,0x9bd8,0xabbb,0xbb9a,0x4a75,0x5a54,0x6a37,0x7a16,
	0x0af1,0x1ad0,0x2ab3,0x3a92,0xfd2e,0xed0f,0xdd6c,0xcd4d,0xbdaa,0xad8b,0x9de8,
	0x8dc9,0x7c26,0x6c07,0x5c64,0x4c45,0x3ca2,0x2c83,0x1ce0,0x0cc1,0xef1f,0xff3e,
	0xcf5d,0xdf7c,0xaf9b,0xbfba,0x8fd9,0x9ff8,0x6e17,0x7e36,0x4e55,0x5e74,0x2e93,
	0x3eb2,0x0ed1,0x1ef0
	};

// ===========================================================================
// CSISWriter
// Holds SIS file info, as generated by the parser and builds the output SIS file.
// All strings are held internally by this class as UNICODE (wide) strings.
// ===========================================================================

CSISWriter::CSISWriter()
:m_dwUID(0x00000000),
m_wMajor(0),
m_wMinor(0),
m_dwBuild(0),
m_wType(EInstSISApp),
m_wFlags(0),
m_wNumLangs(0),
m_pLangStringBase(NULL),
m_pLangBase(NULL),
m_pPkgLineBase(NULL),
m_pDependBase(NULL),
m_pSignatureBase(NULL),
m_pCapabilityBase(NULL),
m_bCmdPassword(FALSE),
m_pObserver(NULL)

	{ ; }

void CSISWriter::Release()
// Dispose of components resources
	{
	// remove any temporary files
	CleanupTemp();
	
	// free up allocated structures
	LANGSTRINGNODE *pLangStringNode = m_pLangStringBase;
	while(m_pLangStringBase)
		{
		pLangStringNode = pLangStringNode->pNext;
		delete m_pLangStringBase;
		m_pLangStringBase = pLangStringNode;
		}
	
	LANGNODE *pLangNode = m_pLangBase;
	while(m_pLangBase)
		{
		pLangNode = pLangNode->pNext;
		delete m_pLangBase;
		m_pLangBase = pLangNode;
		}
	
	PKGLINENODE *pPkgLineNode = m_pPkgLineBase;
	while(m_pPkgLineBase)
		{
		pPkgLineNode = m_pPkgLineBase->pNext;
		switch (m_pPkgLineBase->iPackageLineType)
			{
			case EInstPkgLineFile:
			case EInstPkgLineLanguageFile:
				delete m_pPkgLineBase->file;
				break;
			case EInstPkgLineOption:
				LANGSTRINGNODE *pLangStringNode2;
				pLangStringNode = m_pPkgLineBase->option->pLangStringBase;
				while(pLangStringNode)
					{
					pLangStringNode2 = pLangStringNode->pNext;
					delete pLangStringNode;
					pLangStringNode = pLangStringNode2;
					}
				delete m_pPkgLineBase->option;
				break;
			case EInstPkgLineCondIf:
			case EInstPkgLineCondElseIf:
				ReleaseCondExpr(m_pPkgLineBase->cond);
				break;
			case EInstPkgLineCondElse:
			case EInstPkgLineCondEndIf:
				break;
			}
		delete m_pPkgLineBase;
		m_pPkgLineBase = pPkgLineNode;
		}
	
	DEPENDNODE *pDependNode = m_pDependBase;
	while(m_pDependBase)
		{
		pDependNode = pDependNode->pNext;
		
		pLangStringNode = m_pDependBase->pLangStringBase;
		while(m_pLangStringBase)
			{
			pLangStringNode = pLangStringNode->pNext;
			delete m_pLangStringBase;
			m_pLangStringBase = pLangStringNode;
			}
		delete m_pDependBase;
		m_pDependBase = pDependNode;
		}
	
	if (m_pSignatureBase) delete m_pSignatureBase;
	
	CAPABILITYNODE *pCapabilityNode = m_pCapabilityBase;
	while(m_pCapabilityBase)
		{
		pCapabilityNode = pCapabilityNode->pNext;
		delete m_pCapabilityBase;
		m_pCapabilityBase = pCapabilityNode;
		}
	
	}

void CSISWriter::ReleaseCondExpr(PKGLINECONDITION* expr)
// recursively called to free up condition expression
	{
	if (expr)
		{
		switch (expr->exprType)
			{
			case EInstCondBinOpEq:
			case EInstCondBinOpNe:
			case EInstCondBinOpGt:
			case EInstCondBinOpLt:
			case EInstCondBinOpGe:
			case EInstCondBinOpLe:
			case EInstCondLogOpAnd:
			case EInstCondLogOpOr:
				ReleaseCondExpr(expr->b.pLhs);
				ReleaseCondExpr(expr->b.pRhs);
				break;
			case EInstCondUnaryOpNot:
				ReleaseCondExpr(expr->pExpr);
				break;
			case EInstCondFuncAppCap:
				ReleaseCondExpr(expr->pArg[0]);
				ReleaseCondExpr(expr->pArg[1]);
				break;
			case EInstCondFuncExists:
			case EInstCondFuncDevCap:
				ReleaseCondExpr(expr->pArg[0]);
				break;
			case EInstCondPrimTypeString:
			case EInstCondPrimTypeVariable:
			case EInstCondPrimTypeNumber:
				delete expr->pPrim;
				break;
			default:
				break;
			}
		delete expr;
		}
	}

void CSISWriter::SetCmdPassword(LPCWSTR pszCmdPassword)
	{
	wcscpy(m_pszCmdPassword, pszCmdPassword);
	m_bCmdPassword = TRUE;
	}

BOOL CSISWriter::GetCmdPassword(LPWSTR pszCmdPassword) const
	{
	wcscpy(pszCmdPassword, m_pszCmdPassword);
	
	return m_bCmdPassword;
	}

void CSISWriter::WriteSIS(LPCWSTR pszTargetFile, BOOL fIsStub /*=FALSE*/)
// Construct the output SIS file
// Inputs   : hFile	   The file handle to write to
//			  fIsStub  IS this a stub SIS file (stub SIS files have no CODE block)
	{
	if(m_wFlags & EInstIsUnicode)
		{
		CSISFileGeneratorT<unsigned short> gen(this);
		gen.GenerateSISFile(pszTargetFile, fIsStub);
		}
	else
		{
		CSISFileGeneratorT<char> gen(this);
		gen.GenerateSISFile(pszTargetFile, fIsStub);
		}
	}

void CSISWriter::AddLanguageNode(LANGNODE *pLNode)
// Adds a language node
// Inputs   : pLNode   The node to add
	{
	pLNode->pNext = m_pLangBase;
	m_pLangBase = pLNode;
	m_wNumLangs++;
	}

void CSISWriter::AddLangStringNode(LANGSTRINGNODE *pLSNode)
// Adds a language string node
// Inputs   : pLSNode   The node to add
	{
	pLSNode->pNext = m_pLangStringBase;
	m_pLangStringBase = pLSNode;
	}

void CSISWriter::AddPkgLineNode(PKGLINENODE *pNode)
// Adds a PKG line node
// Inputs   : pNode   The node to add
	{
	pNode->pNext = m_pPkgLineBase;
	m_pPkgLineBase = pNode;
	}

void CSISWriter::AddDependencyNode(DEPENDNODE *pNode)
// Does exactly what it says on the tin!
// Inputs   : pNode   The node to add
	{
	pNode->pNext = m_pDependBase;
	m_pDependBase = pNode;
	}

void CSISWriter::AddSignatureNode(SIGNATURENODE *pSig)
// Sets the digital signature
// Inputs   : pSig   The node to add
	{
	m_pSignatureBase = pSig;
	}

void CSISWriter::AddCapabilityNode(CAPABILITYNODE *pNode)
// Does exactly what it says on the tin!
// Inputs   : pNode   The node to add
	{
	pNode->pNext = m_pCapabilityBase;
	m_pCapabilityBase = pNode;
	}

BOOL CSISWriter::AreLangStringInit()
// Is there a language string node defined? (i.e. has the pkg language header line been parsed?)
	{
	return (m_pLangStringBase!=NULL);
	}

void CSISWriter::SetVersionInfo(DWORD dwUID, WORD wMajor, WORD wMinor, DWORD dwBuild, TSISType wType, WORD wFlags)
// Inputs   : dwUID							Installee app's uid
//			  dwMajor, dwMinor, dwBuild		The version numbers
//		      wType							SIS file type
//		      wFlags						Installation flags
	{
	m_dwUID		= dwUID;    
	m_wMajor	= wMajor;  
	m_wMinor	= wMinor;  
	m_dwBuild	= dwBuild;
	m_wType		= wType;
	m_wFlags	= wFlags;  
	}

void CSISWriter::QueryVersionInfo (WORD *wMajor, WORD *wMinor, DWORD *dwBuild) const
// Inputs   : dwMajor, dwMinor, dwBuild	The destinations for the version numbers
	{
	*wMajor   = m_wMajor;
	*wMinor   = m_wMinor;
	*dwBuild   = m_dwBuild;
	}

void CSISWriter::SetLangDependFileDestinations(LPCWSTR pszDest, LPCWSTR pszMimeType, DWORD dwType, DWORD dwOptions)
// Purpose  : Patches the last 'numlang' files with a target path and flags (including the
//			  OPTFILE_LANGUAGEDEPENDENT flag)
// Inputs   : pszDest	The target path
//          : pszMimeType The mime type (if relevant)
//			  dwOptions Any specified flags
	{
	PKGLINENODE *pNode = m_pPkgLineBase;
	WORD wNumLangs = 0;
	
	while(pNode && (wNumLangs++ < m_wNumLangs))
		{
		wcscpy(pNode->file->pszDest, pszDest);
		pNode->iPackageLineType = EInstPkgLineLanguageFile;
		pNode->file->type = (TInstFileType)dwType;
		switch (pNode->file->type)
			{
			case EInstFileTypeText:
				pNode->file->options.iTextOption=(TInstFileTextOption)dwOptions;
				break;
			case EInstFileTypeRun:
				pNode->file->options.iRunOption=(TInstFileRunOption)dwOptions;
				break;
			case EInstFileTypeMime:
				pNode->file->options.iMimeOption=(TInstFileMimeOption)dwOptions;
				wcscpy(pNode->file->pszMimeType, pszMimeType);
				break;
			default:
				break;
			}
		pNode = pNode->pNext;
		}
	}

void CSISWriter::SetLangDependCompDestinations(DWORD dwUID)
// Purpose  : Sets the component id of the last 'numlang' file nodes, and sets their flags to
//			  OPTFILE_LANGUAGEDEPENDENT | OPTFILE_COMPONENT;
// Inputs   : dwUID  The components UID
	{
	PKGLINENODE *pNode = m_pPkgLineBase;
	WORD wNumLangs = 0;
	
	while(pNode && (wNumLangs++ < m_wNumLangs))
		{
		pNode->iPackageLineType = EInstPkgLineLanguageFile;
		pNode->file->type = EInstFileTypeComponent;
		pNode->file->options.iComponentUid = dwUID;
		pNode = pNode->pNext;
		}
	}

// ===========================================================================
// CSISFileGeneratorBase
// Handles most of the SIS file generation operation. Defers the charater
// width specific stuff to a derived template class...
// ===========================================================================

CSISFileGeneratorBase::CSISFileGeneratorBase(const CSISWriter *pSISWriter)
:m_pSISWriter(pSISWriter)
	{ ; }

CSISFileGeneratorBase::~CSISFileGeneratorBase()
	{ ; }

void CSISFileGeneratorBase::GenerateSISFile(LPCWSTR pszTargetFile, BOOL fIsStub /* = FALSE */)
// Purpose  : Construct
// Inputs   : hFile	   The file handle to write to
//			  fIsStub  IS this a stub SIS file (stub SIS files have no CODE block)
// Returns  : Success or failure (actually TRUE or throws an EGeneratorException
	{
	m_hFile = ::MakeSISOpenFile(pszTargetFile, GENERIC_WRITE, CREATE_ALWAYS);
	if(m_hFile == INVALID_HANDLE_VALUE) throw ErrCannotWriteFile;
	m_stubFile=fIsStub;
	// convert any text files to unicode if necessary
	if (SizeOfChar()!=1 && !m_stubFile) CheckUnicodeTextFiles();
	
	// Estimate the maximum amount of space an installation could require
	
	EstimateInstallationSizeL();
	
	// Compress files into temp directory
	CompressFiles();
	
	//Re-Write Certificate files
	ParseCertificates();
	
	// Calculate the offsets
	CalculateOffsets();
	
	m_wCheck = 0;	// reset checksum
	
	WriteHeaderL();
	WriteLanguageL();
	WritePkgLinesL();
	WriteDependL();
	WriteCertificatesL();
	WriteCapabilitiesL();
	WriteStringsL();
	if(!m_stubFile) WriteCodeL();
	WriteCrcL();
	::CloseHandle(m_hFile);
	WriteSignatureL(pszTargetFile);
	}

void CSISFileGeneratorBase::CheckUnicodeTextFiles()
// Converts any text files to UNICODE
	{
	Verbage(__T("Converting text files to unicode"));
	
	PKGLINENODE *pNode = (PKGLINENODE *)m_pSISWriter->GetPkgLineBase();
	BOOL converted=FALSE;
	
	while(pNode)
		{
		if (pNode->iPackageLineType==EInstPkgLineFile || pNode->iPackageLineType==EInstPkgLineLanguageFile)
			{
			WORD wMaxLangs=1;
			if (pNode->iPackageLineType==EInstPkgLineLanguageFile) wMaxLangs = m_pSISWriter->GetNoLanguages();
			WORD wNumLang = 0;
			
			while(pNode && (wNumLang < wMaxLangs))
				{
				if(pNode->file->type==EInstFileTypeText)
					{	// have a text file
					BOOL littleEndian;
					try
						{
						if (!FileIsUnicode(pNode->file->pszSource,littleEndian))
							{
							LPWSTR pszTempSource=ConvertFileToUnicode(pNode->file->pszSource);
							wcscpy(pNode->file->pszSource, pszTempSource);
							converted=TRUE;
							}
						else if (!littleEndian)
							{
							LPWSTR pszTempSource=ConvertFileToLittleEndianUnicode(pNode->file->pszSource);
							wcscpy(pNode->file->pszSource, pszTempSource);
							converted=TRUE;
							}
						}
					catch (TUtilsException excp)
						{
#ifdef _UNICODE
						  m_pSISWriter->GetObserver()->DoErrMsg((const _TCHAR*)pNode->file->pszSource); // @todo
#else
						DWORD length=wcslen(pNode->file->pszSource);
						LPSTR fName=MakeMBCSString(pNode->file->pszSource, CP_ACP, length);
						m_pSISWriter->GetObserver()->DoErrMsg((const _TCHAR*)fName);
						delete [] fName;
#endif
						throw excp;
						}
					HANDLE hInFile = ::MakeSISOpenFile(pNode->file->pszSource, GENERIC_READ, OPEN_EXISTING);
					if(hInFile == INVALID_HANDLE_VALUE)
						throw ErrCannotOpenFile;				
					if (converted) pNode->file->dwSize=::GetFileSize(hInFile, NULL);
					::CloseHandle(hInFile);
					}
				pNode = pNode->pNext;
				wNumLang++;			
				}
			}
		else
			pNode = pNode->pNext;
		}
	}

void CSISFileGeneratorBase::CalculateOffsets()
// Calculates the offsets of the various component blocks from the head of the file
	{
	assert(m_pSISWriter);
	// Calculate the language offset (The size of the HEADER block) -------------------------------
	m_dwOffLang = sizeof(TInstInstallation);
	
	// Calculate the PKG lines offset (The size of the LANGUAGE	block) --------------------------------
	m_dwOffPkgLines = m_dwOffLang + (sizeof(TInstLang) * m_pSISWriter->GetNoLanguages());
	
	// Calculate the dependency offset (The size of the PKG lines block) ------------------------------
	m_dwOffDepend = m_dwOffPkgLines;
	DWORD stringSizes=0;
	const PKGLINENODE *pNode = m_pSISWriter->GetPkgLineBase(); 
	while(pNode)
		{
		m_dwOffDepend += sizeof(uint32_t);
		switch (pNode->iPackageLineType)
			{
			case EInstPkgLineFile:
			case EInstPkgLineLanguageFile:
				{
				WORD wNumLangs = 1;
				if (pNode->iPackageLineType==EInstPkgLineLanguageFile)
					wNumLangs = m_pSISWriter->GetNoLanguages();
				
				m_dwOffDepend += sizeof(TInstFile) +
					((sizeof(uint32_t) + sizeof(FOFF) + sizeof(uint32_t)) * wNumLangs) +
					sizeof(TInstString);
				stringSizes += (wcslen(pNode->file->pszSource) + wcslen(pNode->file->pszDest)) * SizeOfChar() +
					wcslen(pNode->file->pszMimeType);
				
				while(pNode && (wNumLangs > 1))
					{
					pNode = pNode->pNext;
					wNumLangs--;
					}
				}
				break;
			case EInstPkgLineOption:
				{
				m_dwOffDepend += sizeof(TInstOptions); 
				const LANGSTRINGNODE* pLSNode = pNode->option->pLangStringBase;
				while(pLSNode)
					{
					m_dwOffDepend += sizeof(TInt32)+sizeof(FOFF);
					stringSizes += wcslen(pLSNode->pszString) * SizeOfChar();
					pLSNode = pLSNode->pNext;
					}
				m_dwOffDepend += sizeof(TUint32)*KInstallMaxOptionsInts; 
				}
				break;
			case EInstPkgLineCondIf:
			case EInstPkgLineCondElseIf:
				m_dwOffDepend += sizeof(TInt32) + CalcConditionExprSize(pNode->cond, &stringSizes);
				break;
			case EInstPkgLineCondElse:
			case EInstPkgLineCondEndIf:
				break;
			}
		pNode = pNode->pNext;
		}
	
	// Calculate the certificate block offset (the size of The DEPENDENCY block) -----------------------------
	m_dwOffCertificates = m_dwOffDepend;
	const DEPENDNODE *pDNode = m_pSISWriter->GetDependencyBase(); 
	while(pDNode)
		{
		m_dwOffCertificates += sizeof(TInstDependency) + ((sizeof(DWORD) + sizeof(FOFF)) * m_pSISWriter->GetNoLanguages());
		pDNode = pDNode->pNext;
		}
	
	// Calculate capabilities offsets (the size of the certificates block)
	DWORD size=0;
#ifdef DIGITAL_SIGNATURES
	const SIGNATURENODE* pSignature = m_pSISWriter->GetSignatureBase();
	HANDLE hInFile;
	if (pSignature)
		{
		if (m_bWriteChain)
			{
			hInFile = ::MakeSISOpenFile(pSignature->pszChainFile, GENERIC_READ, OPEN_EXISTING);	
			if (hInFile == INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
			}
		else
			{
			//write certificate verbatim
			hInFile = ::MakeSISOpenFile(pSignature->pszB64File, GENERIC_READ, OPEN_EXISTING);	
			if (hInFile == INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
			}
		size=::GetFileSize(hInFile, NULL) + sizeof (TInstCertBlock);
		
		::CloseHandle(hInFile);
		if (size==0xFFFFFFFF) throw ErrCannotOpenFile;
		}
#endif
	m_dwOffCapabilities = m_dwOffCertificates+size;
	
	// Calculate the string block offset (the size of the CAPABILITIES block) -----------------------------
	m_dwOffString = m_dwOffCapabilities;
	const CAPABILITYNODE* pCNode = m_pSISWriter->GetCapabilityBase(); 
	while(pCNode)
		{
		m_dwOffString += sizeof(TInstCapability);
		pCNode = pCNode->pNext;
		}
	
	// Calculate the code offset (the size of the STRING block) -----------------------------------
	// 1. The Language Strings
	m_dwOffCode = m_dwOffString;
	m_dwOffCode += (sizeof(DWORD) + sizeof(FOFF)) * m_pSISWriter->GetNoLanguages();
	
	const LANGSTRINGNODE *pLSNode = m_pSISWriter->GetLangStringBase();
	while(pLSNode)
		{
		m_dwOffCode += wcslen(pLSNode->pszString) * SizeOfChar();
		pLSNode = pLSNode->pNext;
		}	
	// 2. Allow for any package line strings
	m_dwOffCode += stringSizes;
	// 3. Any File dependencies 
	pDNode = m_pSISWriter->GetDependencyBase();
	while(pDNode)
		{
		pLSNode = pDNode->pLangStringBase;
		while(pLSNode)
			{
			m_dwOffCode += wcslen(pLSNode->pszString) * SizeOfChar();
			pLSNode = pLSNode->pNext;
			}
		pDNode = pDNode->pNext;
		}
	// Calculate the signature offset (the size of the CODE block) -----------------------------------
	m_dwOffSignature=m_dwOffCode;
	pNode = m_pSISWriter->GetPkgLineBase();
	while(pNode)
		{
		if((pNode->iPackageLineType == EInstPkgLineFile || pNode->iPackageLineType == EInstPkgLineLanguageFile) &&
			pNode->file->type != EInstFileTypeNull)
			{
			m_dwOffSignature+=pNode->file->dwSize;
			}
		pNode = pNode->pNext;
		}
	}
	
DWORD CSISFileGeneratorBase::CalcConditionExprSize(const PKGLINECONDITION* expr, DWORD* stringSizes)
		{
		DWORD size=0;
		if (expr)
			{
			size=sizeof(TInstCondExpr);
			switch (expr->exprType)
				{
				case EInstCondBinOpEq:
				case EInstCondBinOpNe:
				case EInstCondBinOpGt:
				case EInstCondBinOpLt:
				case EInstCondBinOpGe:
				case EInstCondBinOpLe:
				case EInstCondLogOpAnd:
				case EInstCondLogOpOr:
					size+=CalcConditionExprSize(expr->b.pLhs,stringSizes)+CalcConditionExprSize(expr->b.pRhs,stringSizes);
					break;
				case EInstCondUnaryOpNot:
					size+=CalcConditionExprSize(expr->pExpr,stringSizes);
					break;
				case EInstCondFuncAppCap:
					size+=CalcConditionExprSize(expr->pArg[0],stringSizes);
					size+=CalcConditionExprSize(expr->pArg[1],stringSizes);
					break;
				case EInstCondFuncDevCap:
				case EInstCondFuncExists:
					size+=CalcConditionExprSize(expr->pArg[0],stringSizes);
					break;
				case EInstCondPrimTypeString:
					*stringSizes+=wcslen(expr->pPrim->pszString) * SizeOfChar();
				case EInstCondPrimTypeVariable:
				case EInstCondPrimTypeNumber:
					size+=sizeof(TInstCondPrim);
					break;
				default:
					break;
				}
			}
		return size;
		}
	
void CSISFileGeneratorBase::WriteHeaderL()
  // Purpose  : Writes the SIS file HEADER block                   
  // Inputs   : fIsStub - TRUE if writing a stub file
{
	Verbage(__T("Writing installation header"));

	TInstInstallation iiHead;

	// Clear the header
	memset((void *)&iiHead, '\0', sizeof(TInstInstallation));

	// Set the UID
	iiHead.iThisUID.iUid[0] = m_pSISWriter->GetUID();
		
	if(m_pSISWriter->GetFlags() &  EInstIsUnicode)
		iiHead.iThisUID.iUid[1] = KUidInstallAppUid1U;
	else
		iiHead.iThisUID.iUid[1] = KUidInstallAppUid1;		 // Narrow output

	iiHead.iThisUID.iUid[2] = KUidInstallAppValue;
		
	// Write the UID Checksum

	flipTInstUID(iiHead.iThisUID); // Flip it first

	DWORD dwCheck = 0;
	UIDCheck(((char *)&iiHead.iThisUID.iUid[0]) + 1);
	dwCheck = (DWORD)(m_wCheck) << 16;
	m_wCheck = 0;
	UIDCheck((char *)&iiHead.iThisUID.iUid[0]);
	dwCheck |= m_wCheck;



	iiHead.iUidChecksum = dwCheck;
		
	// CRC
	m_wCheck = 0;
	Crc((void *)&iiHead, sizeof (TInstUID));

	flipi(iiHead.iUidChecksum); // Always flip before Crc

	Crc((void *)&iiHead.iUidChecksum, sizeof (DWORD)); 

	flipTInstUID(iiHead.iThisUID); // Then flip back so that below it is ok.		
	flipi(iiHead.iUidChecksum);  

	// Set the version
	iiHead.iInstVersion = KInstallerVersion;
		
	// Set the type & flags
	iiHead.iType = (TInt16)m_pSISWriter->GetType();
	iiHead.iFlags = m_pSISWriter->GetFlags();
		
	// Set this version
	WORD wMajor, wMinor;
	DWORD dwBuild;
	m_pSISWriter->QueryVersionInfo(&wMajor, &wMinor, &dwBuild);
		
	iiHead.iCompVersion.iMajor		= (WORD)wMajor;
	iiHead.iCompVersion.iMinor		= (WORD)wMinor;
	iiHead.iCompVersion.iBuild		= dwBuild;
		
	// Set the languages
	iiHead.iNumLanguages   = m_pSISWriter->GetNoLanguages();
	iiHead.iLanguageBlock = m_dwOffLang;
		
	// Set the PKG lines
	const PKGLINENODE *pNode = m_pSISWriter->GetPkgLineBase();
	while(pNode) {
		iiHead.iNumPkgLines++;			
		if (pNode->iPackageLineType==EInstPkgLineFile ||
		    pNode->iPackageLineType==EInstPkgLineLanguageFile) {
			if (pNode->file->pszDest[0]) {
				if (iiHead.iInstallDrive==0)
					iiHead.iInstallDrive=pNode->file->pszDest[0];
				else if (iiHead.iInstallDrive!=pNode->file->pszDest[0])
					iiHead.iInstallDrive = '!';
			}
			if (pNode->iPackageLineType==EInstPkgLineLanguageFile) {
				WORD wMaxLangs = m_pSISWriter->GetNoLanguages();
				WORD wNumLang = 0;
				while(pNode && (wNumLang < wMaxLangs-1)) {
					pNode = pNode->pNext;
					wNumLang++;
				}
			}
		}
		pNode = pNode->pNext;
	}
	iiHead.iPackageLinesBlock = m_dwOffPkgLines;
		
	// Set the dependencies
	const DEPENDNODE *pDNode = m_pSISWriter->GetDependencyBase();
	while(pDNode) {
		iiHead.iNumDependencies++;
		pDNode = pDNode->pNext;
	}
	iiHead.iDependenciesBlock = m_dwOffDepend;
		
	// set the certificates block offset
#ifdef DIGITAL_SIGNATURES
	if (m_pSISWriter->GetSignatureBase())
		iiHead.iCertificatesBlock = m_dwOffCertificates;
#endif
		
	// Set the capabilities
	const CAPABILITYNODE *pCNode = m_pSISWriter->GetCapabilityBase();
	while(pCNode) {
		iiHead.iNumCapabilities++;
		pCNode = pCNode->pNext;
	}
	iiHead.iCapabilitiesBlock = m_dwOffCapabilities;
		
	// Set the Names
	iiHead.iNames = m_dwOffString;
	m_dwCurrString = m_dwOffString + ((sizeof(DWORD) + sizeof(FOFF)) * m_pSISWriter->GetNoLanguages());
		
	// Check if we are writing a stub file
	if(m_stubFile) {
		iiHead.iInstallDrive = (WORD)'z';
		iiHead.iNumPkgLinesDone = iiHead.iNumPkgLines;
		iiHead.iLangChosen = 0;
	}
		
	// set maximum size required to install
	iiHead.iMaximumSpaceNeeded = m_dwMaximumSpaceNeeded;
		
	// Set the digital signature offset
#ifdef DIGITAL_SIGNATURES
	if (m_pSISWriter->GetSignatureBase())
		iiHead.iSignatureBlock = m_dwOffSignature;
#endif
	flipTInstInstallation(iiHead); // byte re-order before write.

	// Write the buffer
	DWORD dwNumBytes;
	if(!::WriteFile(m_hFile, (LPVOID)&iiHead, sizeof(TInstInstallation), &dwNumBytes, NULL))
		throw ErrFailedToWriteHeader;
		
	m_dwCurrCode = m_dwOffCode;
		
	// CRC
	Crc((void *)(&iiHead.iNumLanguages), sizeof (TInstInstallation) - sizeof(TInstUID) - sizeof(DWORD) - sizeof(WORD));
		
}

void CSISFileGeneratorBase::WriteLanguageL()
// To write the SIS file's LANGUAGES block
	{
	Verbage(__T("Writing languages"));
	
	DWORD dwLangCount      = m_pSISWriter->GetNoLanguages();
	const LANGNODE *pLNode = m_pSISWriter->GetLanguageBase();
	WORD *pLang            = new WORD[dwLangCount];
	if (!pLang) throw ErrNotEnoughMemory;
	
	WORD wCount = 0;
	while(pLNode)
		{
		pLang[wCount++] = pLNode->wLang;
		pLNode = pLNode->pNext;
		}
	
	pflips(pLang,dwLangCount); // byte re-order as required

	DWORD dwNumBytes;
	if(!::WriteFile(m_hFile, (LPVOID)pLang, sizeof(WORD) * dwLangCount, &dwNumBytes, NULL))
		throw ErrFailedToWriteLanguages;
	
	Crc((void *)pLang, sizeof(WORD) * dwLangCount);
	
	delete [] pLang;
	}

void CSISFileGeneratorBase::WritePkgLinesL()
// To write the SIS pkg lines block
	{
	Verbage(__T("Writing PKG lines block"));
	
	DWORD dwNumBytes;
	const PKGLINENODE *pNode = m_pSISWriter->GetPkgLineBase();
	
	while(pNode)
		{
		  DWORD plt = pNode->iPackageLineType;
		  flipi(plt);
		  // write type
		  if(!::WriteFile(m_hFile, (LPVOID)&plt, sizeof(TInstPackageLineType), &dwNumBytes, NULL))
		    throw ErrFailedToWriteFilesBlock;		
		  Crc((void *)&plt, sizeof(TInstPackageLineType));
		  // write specific PKG node data
		  switch (pNode->iPackageLineType)
		    {
		    case EInstPkgLineFile:
		    case EInstPkgLineLanguageFile:
		      WriteFileNodeL(pNode);
		      if (pNode->iPackageLineType==EInstPkgLineLanguageFile)
			{
			  WORD wMaxLangs = m_pSISWriter->GetNoLanguages();
			  WORD wNumLang = 0;
			  while(pNode && (wNumLang < wMaxLangs-1))
			    {
			      pNode = pNode->pNext;
			      wNumLang++;
			    }
			}
		      break;
		    case EInstPkgLineOption:
		      WriteOptionsNodeL(pNode);
		      break;
		    case EInstPkgLineCondIf:
		    case EInstPkgLineCondElseIf:
		      {
			DWORD stringSizes;
			DWORD size=CalcConditionExprSize(pNode->cond, &stringSizes);
			flipi(size); // re-order
			if(!::WriteFile(m_hFile, (LPVOID)&size, sizeof(DWORD), &dwNumBytes, NULL))
			  throw ErrFailedToWriteFilesBlock;		
			Crc((void *)&size, sizeof(DWORD));
			WriteConditionNodeL(pNode);
		      }
		      break;
		    case EInstPkgLineCondElse:
		    case EInstPkgLineCondEndIf:
		      break;
		    }
		  pNode = pNode->pNext;
		}
	}

void CSISFileGeneratorBase::WriteFileNodeL(const PKGLINENODE* pNode)
// To write the SIS files FILES block
	{
	const PKGLINEFILE* pFNode=pNode->file;
	DWORD dwNumBytes;
	TInstFile ifFile;
	
	memset((void *)&ifFile, '\0', sizeof(TInstFile));
	ConfigureFileNode(pFNode, &ifFile);
	
	flipTInstFile(ifFile); // byte re-order
	if(!::WriteFile(m_hFile, (LPVOID)&ifFile, sizeof(ifFile), &dwNumBytes, NULL))
		throw ErrFailedToWriteFilesBlock;
	
	// CRC
	Crc((void *)&ifFile, sizeof(ifFile));
	
	// Set the file lengths and offsets
	const PKGLINENODE *pTempNode = pNode;
	WORD wMaxLangs = 1;
	if (pNode->iPackageLineType==EInstPkgLineLanguageFile) wMaxLangs = m_pSISWriter->GetNoLanguages();
	
	// write file lengths (compressed lengths if enabled)
	WORD wNumLang = 0;
	while(pNode && (wNumLang < wMaxLangs))
		{
		  DWORD fdwsize = pNode->file->dwSize;
		  flipi(fdwsize); // Re-order
		  if (!::WriteFile(m_hFile, (LPVOID)&fdwsize, sizeof(DWORD), &dwNumBytes, NULL))
			throw ErrFailedToWriteFilesBlock;
		
		// CRC
		Crc((void *)&fdwsize, sizeof(DWORD));
		
		pNode = pNode->pNext;
		wNumLang++;		
		}
	
	// write file offsets
	pNode = pTempNode;
	wNumLang = 0;
	while(pNode && (wNumLang < wMaxLangs))
	  {
	    DWORD cc = m_dwCurrCode;
	    flipi(cc);
	    if(!::WriteFile(m_hFile, (LPVOID)&cc, sizeof(DWORD), &dwNumBytes, NULL))
	      throw ErrFailedToWriteFilesBlock;
	    // CRC
	    Crc((void *)&cc, sizeof(DWORD));
	    
	    m_dwCurrCode += pNode->file->dwSize;
	    pNode = pNode->pNext;
	    wNumLang++;
	  }
	
	// write uncompressed file lengths
	pNode = pTempNode;
	wNumLang = 0;
	while(pNode && (wNumLang < wMaxLangs))
	  {
	    DWORD dwusize = pNode->file->dwUncompressedSize;
	    flipi(dwusize); // byte re-order
	    if (!::WriteFile(m_hFile, (LPVOID)&dwusize, sizeof(DWORD), &dwNumBytes, NULL))
	      throw ErrFailedToWriteFilesBlock;
	    
	    Crc((void *)&dwusize, sizeof(DWORD));
	    
	    pNode = pNode->pNext;
	    wNumLang++;		
	  }
	
	// write mime type if appropriate
	TInstString mimeType;
	mimeType.iCount=wcslen(pFNode->pszMimeType);	// NOTE: 8 bit string!
	mimeType.iStringPtr=m_dwCurrString;
	m_dwCurrString += mimeType.iCount;
	
	flipTInstString(mimeType);
	if (!::WriteFile(m_hFile, (LPVOID)&mimeType, sizeof(TInstString), &dwNumBytes, NULL))
		throw ErrFailedToWriteFilesBlock;		
	Crc((void *)&mimeType, sizeof(TInstString));
	}

void CSISFileGeneratorBase::ConfigureFileNode(const PKGLINEFILE *pNode, TInstFile *pifFile)
// Purpose  : Converts a FILENODE to a InstFile structure
// Inputs   : pNode   The filenaode to converts
//			  pifFile  A pointer to the target InstFile structure
	{
	pifFile->iFileType = pNode->type;
	pifFile->iOption = pNode->options;
	
	// Set the source
	pifFile->iSource.iCount = wcslen(pNode->pszSource) * SizeOfChar();
	pifFile->iSource.iStringPtr = m_dwCurrString;
	m_dwCurrString += pifFile->iSource.iCount;
	
	// Set the dest
	pifFile->iDestination.iCount = wcslen(pNode->pszDest) * SizeOfChar();
	pifFile->iDestination.iStringPtr = m_dwCurrString;
	m_dwCurrString += pifFile->iDestination.iCount;
	}

void CSISFileGeneratorBase::WriteOptionsNodeL(const PKGLINENODE* pNode)
// To write an options PKG line
	{
	DWORD i,j;
	DWORD dwNumBytes;
	DWORD dwLength;
	const PKGLINEOPTIONS* pONode=pNode->option;
	DWORD dwNumOptions=pONode->dwNumOptions;
	
	DWORD dwno = dwNumOptions;
	
	flipi(dwno);
	if (!::WriteFile(m_hFile, (LPVOID)&dwno, sizeof(DWORD), &dwNumBytes, NULL))
		throw ErrFailedToWriteOptionsBlock;
	// CRC
	Crc((void *)&dwno, sizeof(DWORD));

	const LANGSTRINGNODE *pLSNode = pONode->pLangStringBase;
	for (i=0;i<dwNumOptions;i++)
		{
		const LANGSTRINGNODE *pLSNode2 = pLSNode;
		for(j=0;j<m_pSISWriter->GetNoLanguages();j++)
			{
			dwLength = wcslen(pLSNode->pszString) * SizeOfChar();			

			flipi(dwLength); // byte re-order.
			if(!::WriteFile(m_hFile, (LPVOID)&dwLength, sizeof(DWORD), &dwNumBytes, NULL))
				throw ErrFailedToWriteOptionsBlock;			
			// CRC
			Crc((void *)&dwLength, sizeof(DWORD));
			pLSNode = pLSNode->pNext;			
			}
		pLSNode = pLSNode2;
		for(j=0;j<m_pSISWriter->GetNoLanguages();j++)
			{
			  DWORD mcs = m_dwCurrString;

			  flipi(mcs); // byte re order
			  if(!::WriteFile(m_hFile, (LPVOID)&mcs, sizeof(DWORD), &dwNumBytes, NULL))
				throw ErrFailedToWriteOptionsBlock;
			// CRC
			Crc((void *)&mcs, sizeof(DWORD));
			m_dwCurrString += wcslen(pLSNode->pszString) * SizeOfChar();
			pLSNode = pLSNode->pNext;
			}
		}
	// write out option selection values
	DWORD dwSelected=0xFFFFFFFF;
	for (i=0;i<KInstallMaxOptionsInts;i++)
		{
		  DWORD dws = dwSelected;
		  flipi(dws);
		if(!::WriteFile(m_hFile, (LPVOID)&dws, sizeof(DWORD), &dwNumBytes, NULL))
			throw ErrFailedToWriteOptionsBlock;			
		// CRC
		Crc((void *)&dws, sizeof(DWORD));
		}
	}

void CSISFileGeneratorBase::WriteConditionNodeL(const PKGLINENODE* pNode)
// To write a condition PKG line
	{
	const PKGLINECONDITION* pCNode=pNode->cond;
	// write out recursive condition expression(s)
	WriteCondExprL(pCNode);	
	}

void CSISFileGeneratorBase::WriteCondExprL(const PKGLINECONDITION* expr)
	{
	TInstCondExpr iceExpr;
	TInstCondPrim prim;
	DWORD dwNumBytes;
	
	if (expr)
		{
		  flipTInstCondExpr(iceExpr);
		  iceExpr.iExprType=expr->exprType;
		  if(!WriteFile(m_hFile, (LPVOID)&iceExpr, sizeof(TInstCondExpr), &dwNumBytes, NULL))
		    throw ErrFailedToWriteConditionBlock;
		  // CRC
		  Crc((void *)&iceExpr, sizeof(TInstCondExpr));
		switch (expr->exprType)
			{
			case EInstCondBinOpEq:
			case EInstCondBinOpNe:
			case EInstCondBinOpGt:
			case EInstCondBinOpLt:
			case EInstCondBinOpGe:
			case EInstCondBinOpLe:
			case EInstCondLogOpAnd:
			case EInstCondLogOpOr:
				WriteCondExprL(expr->b.pLhs);
				WriteCondExprL(expr->b.pRhs);
				break;
			case EInstCondUnaryOpNot:
				WriteCondExprL(expr->pExpr);
				break;
			case EInstCondFuncAppCap:
				WriteCondExprL(expr->pArg[0]);
				WriteCondExprL(expr->pArg[1]);
				break;
			case EInstCondFuncDevCap:
			case EInstCondFuncExists:
				WriteCondExprL(expr->pArg[0]);
				break;
			case EInstCondPrimTypeString:
				{
				prim.iString.iCount = wcslen(expr->pPrim->pszString) * SizeOfChar();
				prim.iString.iStringPtr = m_dwCurrString;
				m_dwCurrString += prim.iString.iCount;

				flipTInstCondPrim(prim); // re order
				if(!WriteFile(m_hFile, (LPVOID)&prim, sizeof(TInstCondPrim), &dwNumBytes, NULL))
					throw ErrFailedToWriteConditionBlock;
				// CRC
				Crc((void *)&prim, sizeof(TInstCondPrim));
				}
				break;
			case EInstCondPrimTypeVariable:
			case EInstCondPrimTypeNumber:
				{
				prim.iNumber=expr->pPrim->dwNumber;

				flipTInstCondPrim(prim); // re order
				if(!WriteFile(m_hFile, (LPVOID)&prim, sizeof(TInstCondPrim), &dwNumBytes, NULL))
					throw ErrFailedToWriteConditionBlock;
				// CRC
				Crc((void *)&prim, sizeof(TInstCondPrim));
				}
				break;
			default:
				break;
			}
		}
	}

void CSISFileGeneratorBase::WriteDependL()
// To write the SIS file's DEPENDENCY block
	{
	Verbage( __T("Writing dependencies"));
	
	const DEPENDNODE *pDNode = m_pSISWriter->GetDependencyBase();
	while(pDNode)
		{
		TInstDependency idDepend;
		
		idDepend.iUid = pDNode->dwUID;
		
		idDepend.iVersion.iMajor    = pDNode->wMajor;
		idDepend.iVersion.iMinor    = pDNode->wMinor;
		idDepend.iVersion.iBuild    = pDNode->dwBuild;
		
		DWORD dwNumBytes;

		flipTInstDependency(idDepend); // re order

		if(!WriteFile(m_hFile, (LPVOID)&idDepend, sizeof(TInstDependency), &dwNumBytes, NULL))
			throw ErrFailedToWriteDependencyBlock;
		
		// CRC
		Crc((void *)&idDepend, sizeof(idDepend));
		
		// Now write out the names
		LANGSTRINGNODE *pLSNode = pDNode->pLangStringBase;
		while(pLSNode)
			{
			DWORD dwLength = wcslen(pLSNode->pszString) * SizeOfChar();
			
			DWORD dwNumBytes;

			flipi(dwLength); // re order

			if(!::WriteFile(m_hFile, (LPVOID)&dwLength, sizeof(DWORD), &dwNumBytes, NULL))
				throw ErrFailedToWriteDependencyBlock;
			
			// CRC
			Crc((void *)&dwLength, sizeof(DWORD));
			pLSNode = pLSNode->pNext;
			}
		
		pLSNode = pDNode->pLangStringBase;
		while(pLSNode)
			{
			DWORD dwNumBytes;
			DWORD mcs = m_dwCurrString;

			flipi(mcs);
			if(!::WriteFile(m_hFile, (LPVOID)&mcs, sizeof(DWORD), &dwNumBytes, NULL))
				throw ErrFailedToWriteDependencyBlock;
			
			// CRC
			Crc((void *)&mcs, sizeof(DWORD));
			
			m_dwCurrString += wcslen(pLSNode->pszString) * SizeOfChar();
			pLSNode = pLSNode->pNext;
			}
		pDNode = pDNode->pNext;
		}
	}

void CSISFileGeneratorBase::WriteCertificatesL()
// To write the SIS file's CERTIFICATES block
	{
#ifdef DIGITAL_SIGNATURES
	//get digital sig info (public key/private key and password)
	const SIGNATURENODE* pSignature = m_pSISWriter->GetSignatureBase();
	
	if (pSignature)
		{
		HANDLE hInFile; 
		DWORD dwFileSize;
		DWORD dwNumBytes;
		
		BYTE *pBuf;
		
		//pck7 file was re-written to a chain file, so read it instead
		if (m_bWriteChain) 
			{
			//open certificate chain
			hInFile = ::MakeSISOpenFile(pSignature->pszChainFile, GENERIC_READ, OPEN_EXISTING);	
			if(hInFile == INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
			
			}
		
		
		//no conversions in pPKCS7 object so write certificate verbatim
		else	
			{
			//open public key file (pre-converted from Base64)
			hInFile = ::MakeSISOpenFile(pSignature->pszB64File, GENERIC_READ, OPEN_EXISTING);	
			if(hInFile == INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
			
			}
		
		
		dwFileSize = ::GetFileSize(hInFile, NULL);
		if (dwFileSize==0xFFFFFFFF) throw ErrCannotOpenFile;
		
		TInstCertBlock CertBlock;
		
		
		//get system time
		struct tm *newtime;
		long ltime;
		time( &ltime );
		/* Obtain GMT */
		newtime = gmtime( &ltime );
		
		//copy RTC to our internal structure
		CertBlock.iTimeStamp.iSecond = (TInt16) newtime->tm_sec;
		CertBlock.iTimeStamp.iMinute = (TInt16) newtime->tm_min;
		CertBlock.iTimeStamp.iHour = (TInt16) newtime->tm_hour;
		
		//date is stored with year 1900 as baseline
		if (newtime->tm_year >= 100)
			{
			newtime->tm_year += Y2K_FIX;
			}
		
		CertBlock.iTimeStamp.iYear = (TInt16) newtime->tm_year;
		CertBlock.iTimeStamp.iMonth = (TInt16) newtime->tm_mon;
		CertBlock.iTimeStamp.iDay = (TInt16) newtime->tm_mday;
		
		CertBlock.iNumCerts = pSignature->iNumCerts;
		
		
		pBuf = new BYTE[dwFileSize + sizeof (TInstCertBlock)];	
		if (!pBuf) throw ErrNotEnoughMemory;
		
		flipTInstCertBlock(CertBlock); // re-order

		memcpy(pBuf, (LPVOID)&CertBlock, sizeof (TInstCertBlock));
		
		BYTE* pBuf2 = &pBuf[sizeof (TInstCertBlock)];
		
		
		if(!::ReadFile(hInFile, (LPVOID)pBuf2, dwFileSize, &dwNumBytes, NULL))
			throw ErrCannotReadFile;
		
		if(!::WriteFile(m_hFile, (LPVOID)pBuf, dwFileSize + sizeof (TInstCertBlock), &dwNumBytes, NULL))
			throw ErrFailedToWriteCertificatesBlock;
		
		// CRC
		Crc((void *)pBuf, dwNumBytes);
		delete [] pBuf;
		
		
		::CloseHandle(hInFile);
		}
#endif
	}

void CSISFileGeneratorBase::WriteCapabilitiesL()
// To write the SIS file's CAPABILITIES block
	{
	Verbage( __T("Writing capabilities"));
	
	const CAPABILITYNODE *pCNode = m_pSISWriter->GetCapabilityBase();
	while(pCNode)
		{
		TInstCapability idCapability;
		
		idCapability.iKey = pCNode->iKey;
		idCapability.iValue = pCNode->iValue;
		
		DWORD dwNumBytes;
		flipTInstCapability(idCapability); // re order

		if(!WriteFile(m_hFile, (LPVOID)&idCapability, sizeof(TInstCapability), &dwNumBytes, NULL))
			throw ErrFailedToWriteCapabilitiesBlock;
		
		// CRC
		Crc((void *)&idCapability, sizeof(TInstCapability));
		
		pCNode = pCNode->pNext;
		}
	}

void CSISFileGeneratorBase::WriteStringsL()
// To write the SIS file's STRINGS block
	{
	Verbage(__T("Writing strings"));
	
	// Write the installation name string descriptors  --------------------------------------------
	const LANGSTRINGNODE *pLSNode = m_pSISWriter->GetLangStringBase();	 
	
	DWORD  dwCount   = m_pSISWriter->GetNoLanguages();
	DWORD* pdwLength = new DWORD [dwCount];
	DWORD* pdwPos    = new DWORD [dwCount];
	DWORD  i = 0;
	
	if (!pdwLength || !pdwPos) throw ErrNotEnoughMemory;
	
	while(pLSNode)
		{
		pdwLength[i] = wcslen(pLSNode->pszString) * SizeOfChar();
		pdwPos[i]    = m_dwCurrString;
		pLSNode	     = pLSNode->pNext;
		m_dwCurrString += pdwLength[i++];
		assert(i <= dwCount);
		}
	
	DWORD dwNumBytes;
	pflipi(pdwLength, dwCount); // re order

	if(!::WriteFile(m_hFile, (LPVOID)pdwLength, dwCount * sizeof(DWORD), &dwNumBytes, NULL))
		throw ErrFailedToWriteStringsBlock;
	Crc((void *)pdwLength, sizeof(DWORD) * dwCount);
	
	pflipi(pdwPos,dwCount); // re order

	if(!::WriteFile(m_hFile, (LPVOID)pdwPos, dwCount * sizeof(DWORD), &dwNumBytes, NULL))
		throw ErrFailedToWriteStringsBlock;
	Crc((void *)pdwPos, sizeof(DWORD) * dwCount);
	
	// Put PKG lines strings ----------------------------------------------------------------
	const PKGLINENODE *pNode = m_pSISWriter->GetPkgLineBase();
	
	while(pNode)
		{
		switch (pNode->iPackageLineType)
			{
			case EInstPkgLineFile:
			case EInstPkgLineLanguageFile:
				{
				WriteString(pNode->file->pszSource, m_hFile);		
				WriteString(pNode->file->pszDest, m_hFile);
				
				// Check for multiple language forms...
				WORD wMaxLangs = 0;
				if (pNode->iPackageLineType==EInstPkgLineLanguageFile) wMaxLangs = m_pSISWriter->GetNoLanguages();
				WORD wNumLang = 0;
				while (pNode && (wNumLang < wMaxLangs-1))
					{
					pNode = pNode->pNext;
					wNumLang++;
					}
				}
				// write mime type
				WriteString8(pNode->file->pszMimeType, m_hFile);		
				break;
			case EInstPkgLineOption:
				{
				pLSNode = pNode->option->pLangStringBase;
				while(pLSNode)
					{
					WriteString(pLSNode->pszString, m_hFile);
					pLSNode=pLSNode->pNext;
					}
				}
				break;
			case EInstPkgLineCondIf:
			case EInstPkgLineCondElseIf:
				WriteCondExprStrL(pNode->cond);
				break;
			case EInstPkgLineCondElse:
			case EInstPkgLineCondEndIf:
				break;
			}
		pNode = pNode->pNext;
		}
	
	// Write the dependency names -----------------------------------------------------------------
	const DEPENDNODE *pDNode = m_pSISWriter->GetDependencyBase();
	while(pDNode)
		{
		// Now write out the names
		pLSNode = pDNode->pLangStringBase;
		while(pLSNode)
			{
			WriteString(pLSNode->pszString, m_hFile);
			pLSNode = pLSNode->pNext;
			}
		pDNode = pDNode->pNext;
		}
	
	// Now write out the installation names -------------------------------------------------------
	pLSNode = m_pSISWriter->GetLangStringBase();	 
	while(pLSNode)
		{
		WriteString(pLSNode->pszString, m_hFile);
		pLSNode = pLSNode->pNext;
		}
	}
	

void CSISFileGeneratorBase::WriteCondExprStrL(const PKGLINECONDITION* expr)
{
	if (expr) {
		switch (expr->exprType) {
		case EInstCondBinOpEq:
		case EInstCondBinOpNe:
		case EInstCondBinOpGt:
		case EInstCondBinOpLt:
		case EInstCondBinOpGe:
		case EInstCondBinOpLe:
		case EInstCondLogOpAnd:
		case EInstCondLogOpOr:
			WriteCondExprStrL(expr->b.pLhs);
			WriteCondExprStrL(expr->b.pRhs);
			break;
		case EInstCondUnaryOpNot:
			WriteCondExprStrL(expr->pExpr);
			break;
		case EInstCondFuncAppCap:
			WriteCondExprStrL(expr->pArg[0]);
			WriteCondExprStrL(expr->pArg[1]);
			break;
		case EInstCondFuncDevCap:
		case EInstCondFuncExists:
			WriteCondExprStrL(expr->pArg[0]);
			break;
		case EInstCondPrimTypeString:
			WriteString(expr->pPrim->pszString, m_hFile);
			break;
		case EInstCondPrimTypeVariable:
		case EInstCondPrimTypeNumber:
			break;
		default:
			break;
		}
	}
}

	
// To write the SIS file's CODE block
void CSISFileGeneratorBase::WriteCodeL()
{
	Verbage(__T("Writing code"));

	const PKGLINENODE *pNode = m_pSISWriter->GetPkgLineBase();
		
	while(pNode) {
		if((pNode->iPackageLineType == EInstPkgLineFile || pNode->iPackageLineType == EInstPkgLineLanguageFile) &&
		   pNode->file->type != EInstFileTypeNull) {
			HANDLE hInFile = ::MakeSISOpenFile(pNode->file->pszSource, GENERIC_READ, OPEN_EXISTING);
			if(hInFile == INVALID_HANDLE_VALUE)
				throw ErrCannotOpenFile;
				
			BYTE *pBuf = new BYTE[pNode->file->dwSize];			
			if (!pBuf) throw ErrNotEnoughMemory;
				
			DWORD dwNumBytes;
			if(!::ReadFile(hInFile, (LPVOID)pBuf, pNode->file->dwSize, &dwNumBytes, NULL))
				throw ErrCannotReadFile;

			if(!::WriteFile(m_hFile, (LPVOID)pBuf, pNode->file->dwSize, &dwNumBytes, NULL))
				throw ErrCannotWriteFile;
				
			// CRC
			Crc((void *)pBuf, pNode->file->dwSize);
			delete [] pBuf;
			::CloseHandle(hInFile);
			_wunlink(pNode->file->pszSource);
		}
		pNode = pNode->pNext;
	}
}

	
// To write the CRC section                           
void CSISFileGeneratorBase::WriteCrcL()
{
	Verbage(__T("Writing crc"));
		
	if(::SetFilePointer(m_hFile, sizeof(TInstUID) + sizeof(DWORD), NULL, FILE_BEGIN) == 0xFFFFFFFF)
		throw ErrFailedToWriteCRC;
		
	DWORD dwNumBytes;
	WORD mcrc = m_wCheck;
	flips(mcrc);
	if(!::WriteFile(m_hFile, (LPVOID)&mcrc, sizeof(WORD), &dwNumBytes, NULL))
		throw ErrFailedToWriteCRC;
}
	

void CSISFileGeneratorBase::WriteSignatureL(LPCWSTR pszTargetFile)
		{
#ifdef DIGITAL_SIGNATURES
		Verbage(__T("Writing digital signature"));
		
		//get private key to encrypt hash for digital signature
		const SIGNATURENODE* pSignature = m_pSISWriter->GetSignatureBase();
		if (pSignature)
			{
			// convert private key file name into MBCS for Encrypt() method
			DWORD length=wcslen(pSignature->pszPrivateKey);
			LPSTR fName=MakeMBCSString(pSignature->pszPrivateKey, CP_ACP, length);
			
			// convert password into MBCS for Encrypt() method
			WCHAR pszCmdPassword[MAX_PATH];	 // Command Line Decryption Password
			LPSTR pszPassword = "";
			
			//test for command line Decryption Password
			if (m_pSISWriter->GetCmdPassword(pszCmdPassword))
				{
				length=wcslen(pszCmdPassword);
				pszPassword=MakeMBCSString(pszCmdPassword, CP_ACP, length);
				
				/*
				#ifdef _DEBUG
				printf("***DEBUG CODE*** Password is %s\n", pszPassword);
				#endif
				*/
				}
			
			else
				{
				length=wcslen(pSignature->pszPassword);
				pszPassword=MakeMBCSString(pSignature->pszPassword, CP_ACP, length);
				}
			
			
			CPrivateKey* pKey = new CPrivateKey;
			if (!pKey) throw ErrNotEnoughMemory;
			
			pKey->LoadKey(fName, pszPassword);
			
			if (pKey->Loaded() == FALSE)
				{	
				throw ErrCannotOpenFile;
				}
			
			
			// compute hash across SIS file
			CHash* pHash = NULL;
			
			pHash = new CSHA1;
			if (!pHash) throw ErrNotEnoughMemory;
			
			char hash[HASHLEN];
			
			// convert SIS file name into MBCS for HashFile() method
			DWORD lengthsis=wcslen(pszTargetFile);
			LPSTR fSISName=MakeMBCSString(pszTargetFile, CP_ACP, lengthsis);
			
			// hash will contain the computed hash for the SIS file
			DWORD TargetLen = pHash->HashFile(fSISName, hash);
			
			
			delete [] fSISName;	
			delete pHash;
			
			//now we need to DSA the SHA-1 signature
			if (pKey->KeyType() == EDSAPrivateKey)
				{
				pHash = new CDSA;
				if (!pHash) throw ErrNotEnoughMemory;
				
				pHash->SetKey(pKey->GetKeyPtr());
				// hash will contain the computed SHA-1 signed by DSA
				pHash->HashFile("", hash);
				
				}
			
			// encrypt hash to create digital signature
			CEncrypt* crypto = new CEncrypt;		
			if (!crypto) throw ErrNotEnoughMemory;
			
			char ciphertext[CIPHERTEXTLEN];
			
			
			
			int KeyLen;
			TKeyCipher ciphertype;
			
			//cipher text will contain the encrypted hash (we only do this for RSA)
			if (pKey->KeyType() == ERSAPrivateKey)
				{
				
				char string2char[20];
				
				pHash->HexStringToBytes(hash, string2char);
				
				if (!crypto->Encrypt(string2char, (unsigned char*)ciphertext, fName, pszPassword, &KeyLen, &ciphertype))
					throw ErrCannotOpenFile;
				}
			
			delete [] pszPassword;
			delete [] fName;
			delete crypto;
			
			// now read in unsigned SIS file & write out new SIS file with appended signature
			
			HANDLE hSISFile = ::MakeSISOpenFile(pszTargetFile, GENERIC_READ, OPEN_EXISTING);
			if(hSISFile == INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
			
			// double check file size is as expected
			if (TargetLen != ::GetFileSize(hSISFile, NULL)) throw ErrFailedToWriteSignatureBlock;
			
			if (pKey->KeyType() == EDSAPrivateKey)
				{
				KeyLen = pKey->KeyLength();
				}
			
			unsigned char* pFileBuf = new unsigned char[TargetLen + KeyLen];
			if (!pFileBuf) throw ErrNotEnoughMemory;
			
			DWORD dwNumBytes;
			if(!::ReadFile(hSISFile, (LPVOID)pFileBuf, TargetLen, &dwNumBytes, NULL))
				throw ErrCannotReadFile;
			
			::CloseHandle(hSISFile);
			
			//append hash cipher text to end of the file
			
			//if RSA, we need to add the RSA'd SHA-1 signature
			if (pKey->KeyType() == ERSAPrivateKey)
				{
				memcpy(&pFileBuf[TargetLen], ciphertext, KeyLen);
				}
			
			//If DSA, we need to add the DSA'd SHA-1 signature
			if (pKey->KeyType() == EDSAPrivateKey)
				{
				memcpy(&pFileBuf[TargetLen], hash, KeyLen);
				
				
				delete pHash;
				}
			
			//write new SIS file to disk (hash is now appended)
			HANDLE hOutFile = ::MakeSISOpenFile(pszTargetFile, GENERIC_WRITE, OPEN_EXISTING);
			if(hOutFile == INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;			
			
			if(!::WriteFile(hOutFile, (LPVOID)pFileBuf, TargetLen + KeyLen, &dwNumBytes, NULL))
				throw ErrCannotWriteFile;
			
			::CloseHandle(hOutFile);
			
			delete [] pFileBuf;
			delete pKey;
			
		}
#endif		
}


// Load the private key from the file
CPrivateKey* CSISFileGeneratorBase::LoadPrivateKey()
	{
	CPrivateKey* result = NULL;
#ifdef DIGITAL_SIGNATURES
	
	const SIGNATURENODE* pSignature = m_pSISWriter->GetSignatureBase();
	if (pSignature)
		{
		// convert private key file name into MBCS for Encrypt() method
		DWORD length=wcslen(pSignature->pszPrivateKey);
		LPSTR fName=MakeMBCSString(pSignature->pszPrivateKey, CP_ACP, length);
		
		// convert password into MBCS for Encrypt() method
		WCHAR pszCmdPassword[MAX_PATH];	 // Command Line Decryption Password
		LPSTR pszPassword = "";
		
		//test for command line Decryption Password
		if (m_pSISWriter->GetCmdPassword(pszCmdPassword))
			{
			length=wcslen(pszCmdPassword);
			pszPassword=MakeMBCSString(pszCmdPassword, CP_ACP, length);
			
			/*
			#ifdef _DEBUG
			printf("***DEBUG CODE*** Password is %s\n", pszPassword);
			#endif
			*/
			}
		
		else
			{
			length=wcslen(pSignature->pszPassword);
			pszPassword=MakeMBCSString(pSignature->pszPassword, CP_ACP, length);
			}
		
		
		CPrivateKey* pKey = new CPrivateKey;
		if (!pKey) throw ErrNotEnoughMemory;
		
		pKey->LoadKey(fName, pszPassword);
		
		delete [] pszPassword;
		delete [] fName;
		
		if (pKey->Loaded() == FALSE)
			{	
			throw ErrCannotOpenFile;
			}
		else
			{
			result = pKey;
			}
		}
#endif		
	return result;
	}



void CSISFileGeneratorBase::Crc(const void *pPtr, DWORD dwLength)
// Purpose  : Creates the CRC checksum                              
// Inputs   : pPtr    - Pointer to a block of memory                 
//            wLength - The length of the memory block            
	{
	const unsigned char *pB = (const unsigned char *)pPtr;
	const unsigned char *pE = pB + dwLength;
	WORD crc = m_wCheck;
    while (pB < pE)
		crc = (WORD)((crc << 8) ^ crcTab[((crc >> 8) ^ *pB++) & 0xff]);
	m_wCheck = crc;
	}

void CSISFileGeneratorBase::UIDCheck(const void *pPtr)
// Purpose  : To create a UID checksum                           
// Inputs   : pPtr    - Pointer to a block of memory                 
	{
	const unsigned char *pB = (const unsigned char *)pPtr;
	const unsigned char *pE = pB + (KUidInstallNumUid * sizeof(DWORD));
	unsigned char buf[(KUidInstallNumUid * sizeof(DWORD))>>1];
	unsigned char *pT = (&buf[0]);
	while (pB < pE)
		{
		*pT++ = (*pB);
		pB += 2;
		} 
	m_wCheck = 0;
	Crc(&buf[0], (KUidInstallNumUid * sizeof(DWORD))>>1);
	}

void CSISFileGeneratorBase::EstimateInstallationSizeL()
	{
	int nodes, i;
	const PKGLINENODE *pNode = m_pSISWriter->GetPkgLineBase();
	
	// This method makes an estimate of the amount of space required by an installation.
	// It sums the uncompressed size of all files that are to be installed on an epoc device.
	// If statements are a special case.  Only the largest file installed by any given branch of
	// an if statement will be added to the total install size.  Consequently the space required will
	// overestimated for package files with if statements. 
	
	// Since nodes are added to the front of the list, the last package line appears at the 
	// front of the list.  Its easier to compute the maximum amount of space required by an
	// if statement if we can access the nodes in the order in which they appear in the 
	// .pkg file.  Hence, an array is created with pointers to these nodes. 
	
	for (nodes = 0; pNode != NULL; pNode = pNode->pNext) 
		nodes++;
	
	const PKGLINENODE **pNodePointers = new const PKGLINENODE*[nodes];
	
	pNode = m_pSISWriter->GetPkgLineBase();
	for (i = nodes-1; i >= 0; i--)
		{
		pNodePointers[i] = pNode;
		pNode = pNode->pNext;
		}
	
	try
		{
		m_dwMaximumSpaceNeeded = 0;
		i = 0;
		while (i < nodes)
			{
			if (pNodePointers[i]->iPackageLineType == EInstPkgLineCondIf)
				m_dwMaximumSpaceNeeded += CalculateInstallSizeOfIfBlockL(pNodePointers, i);
			else
				{
				m_dwMaximumSpaceNeeded += CalculateInstallSizeOfFileL(pNodePointers[i]);
				i++;
				}
			}
		}
	catch (...)
		{
		delete[] pNodePointers;
		throw;
		}
	
	delete[] pNodePointers;
	}

int CSISFileGeneratorBase::CalculateInstallSizeOfIfBlockL(const PKGLINENODE **pNode, int &index) const
	{
	int spaceNeeded = 0;
	int needsOfBlock;
	bool processingIf = true;
	index++;
	
	while (processingIf)
		{
		needsOfBlock = 0;
		while (pNode[index]->iPackageLineType != EInstPkgLineCondElseIf &&
			pNode[index]->iPackageLineType != EInstPkgLineCondElse &&
			pNode[index]->iPackageLineType != EInstPkgLineCondEndIf)
			{
			if (pNode[index]->iPackageLineType == EInstPkgLineCondIf)
				needsOfBlock += CalculateInstallSizeOfIfBlockL(pNode,index);
			else
				{
				needsOfBlock += CalculateInstallSizeOfFileL(pNode[index]);
				index++;
				}
			}
		
		if (needsOfBlock > spaceNeeded)
			spaceNeeded = needsOfBlock;
		
		processingIf = (pNode[index]->iPackageLineType != EInstPkgLineCondEndIf);
		index++;
		}
	
	return spaceNeeded;
	}

int CSISFileGeneratorBase::CalculateInstallSizeOfFileL(const PKGLINENODE *pNode) const
	{
	int spaceNeeded = 0;
	
	if ((pNode->iPackageLineType == EInstPkgLineFile || pNode->iPackageLineType == EInstPkgLineLanguageFile) &&
		(pNode->file->type != EInstFileTypeText && pNode->file->type != EInstFileTypeNull && 
		pNode->file->type != EInstFileTypeMime))
		{
		if (pNode->file->type == EInstFileTypeComponent)
			{
			HANDLE hFile = ::MakeSISOpenFile(pNode->file->pszSource,GENERIC_READ, OPEN_EXISTING);
			if (hFile==INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
			TInstInstallation header;
			DWORD dwBytesRead;
			if (!::ReadFile(hFile,(LPVOID) &header,sizeof(TInstInstallation),&dwBytesRead,NULL))
				throw ErrCannotReadFile;
			::CloseHandle(hFile);
			spaceNeeded = header.iMaximumSpaceNeeded;
			}
		else
			if (pNode->file->type != EInstFileTypeText)
				spaceNeeded = pNode->file->dwSize;
		}
	
	return spaceNeeded;
	}

void CSISFileGeneratorBase::CompressFiles()
// Purpose : To compress files embedded within the sis file.
	{
	int compressedSize = 0;
	
	
	Verbage(__T("Compressing files"));
	const PKGLINENODE *pNode = m_pSISWriter->GetPkgLineBase();
	
	while(pNode)
		{
		if ((pNode->iPackageLineType == EInstPkgLineFile || pNode->iPackageLineType == EInstPkgLineLanguageFile) && pNode->file->type != EInstFileTypeNull)
			{
			pNode->file->dwUncompressedSize = pNode->file->dwSize;
			
			if (((m_pSISWriter->GetFlags() & EInstNoCompress) == 0))
				{
				LPWSTR pszCompressedFileName = TempFileName(pNode->file->pszSource);
				
				CompressFile(pNode,pszCompressedFileName,compressedSize);
				
				// update the source filename and the file size in the PKGLINEFILE node.
				
				wcscpy(pNode->file->pszSource, pszCompressedFileName);
				pNode->file->dwSize = compressedSize;
				}
			}
		pNode = pNode->pNext;
		}
	}


void CSISFileGeneratorBase::CompressFile(const PKGLINENODE *pNode, LPWSTR pszOutputFname, int &compressedSize)
// Purpose: Compresses the file pointed into a temporary file.
// Inputs:  pNode: a pointer to a PKGLINENODE which contains information about the source file
//			pszOutputFname: the name of the compressed file to be created.
// Outputs:	compressedSize: The size of the compressed File.
	{
	HANDLE hUncompressedFile, hCompressedFile;
	DWORD dwUncompressedFileSize;
	const int bufferSize = 0xffff;
	int zlibErr;
	BOOL zlibInit = FALSE;
	
	hUncompressedFile = ::MakeSISOpenFile(pNode->file->pszSource, GENERIC_READ, OPEN_EXISTING);
	if(hUncompressedFile == INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
	dwUncompressedFileSize = ::GetFileSize(hUncompressedFile, NULL);
	
	hCompressedFile = ::MakeSISOpenFile(pszOutputFname, GENERIC_WRITE|GENERIC_READ, CREATE_NEW);
	if (hCompressedFile==INVALID_HANDLE_VALUE) throw ErrCannotOpenFile;
	
	BYTE *pbInputBuffer = new BYTE [bufferSize];
	BYTE *pbOutputBuffer = new BYTE [bufferSize];
	z_stream compressionState;
	
	try
		{
		if (!pbOutputBuffer || !pbInputBuffer) throw ErrNotEnoughMemory;
		
		compressionState.zalloc = NULL;
		compressionState.zfree = NULL;
		compressionState.opaque = NULL;
		compressionState.next_out = pbOutputBuffer;
		compressionState.avail_out = bufferSize;
		ReadFromUncompressedFile(hUncompressedFile,pbInputBuffer,compressionState,bufferSize);
		
		zlibInit = TRUE;
		zlibErr = deflateInit(&compressionState,Z_DEFAULT_COMPRESSION);
		if (zlibErr != Z_OK) throw ErrFailedCompression;
		
		while ((zlibErr = deflate(&compressionState,Z_NO_FLUSH)) == Z_OK)
			{
			if (compressionState.avail_in == 0)
				ReadFromUncompressedFile(hUncompressedFile,pbInputBuffer,compressionState,bufferSize);
			
			if (compressionState.avail_out == 0)
				WriteToCompressedFile(hCompressedFile, pbOutputBuffer,compressionState, bufferSize);
			}
		
		while ((zlibErr = deflate(&compressionState,Z_FINISH)) == Z_OK)
			{
			if (compressionState.avail_in == 0)
				ReadFromUncompressedFile(hUncompressedFile,pbInputBuffer,compressionState,bufferSize);
			
			if (compressionState.avail_out == 0)
				WriteToCompressedFile(hCompressedFile, pbOutputBuffer,compressionState, bufferSize);
			}
		if (zlibErr != Z_STREAM_END) throw ErrFailedCompression;
		WriteToCompressedFile(hCompressedFile, pbOutputBuffer,compressionState, bufferSize);
		} 
	catch(...)
		{
		delete[] pbInputBuffer;
		delete[] pbOutputBuffer;
		::CloseHandle(hUncompressedFile);
		::CloseHandle(hCompressedFile);
		_wunlink(pszOutputFname); // delete compressed file
		if (zlibInit)
			deflateEnd(&compressionState);
		throw;
		}
	
	delete[] pbInputBuffer;
	delete[] pbOutputBuffer;
	::CloseHandle(hUncompressedFile);
	::CloseHandle(hCompressedFile);
	
	compressedSize = compressionState.total_out;
	deflateEnd(&compressionState);	
	}

void CSISFileGeneratorBase::ReadFromUncompressedFile(HANDLE hFile,BYTE *pbBuffer,z_stream &compressionState, int bufferSize)
	{
	DWORD dwNumBytes;
	if (::ReadFile(hFile, (LPVOID)pbBuffer, bufferSize, &dwNumBytes, NULL))
		{
		compressionState.next_in = pbBuffer;
		compressionState.avail_in = dwNumBytes;	
		}
	else
		throw ErrCannotReadFile;
	}

void CSISFileGeneratorBase::WriteToCompressedFile(HANDLE hFile,BYTE *pbBuffer,z_stream &compressionState, int bufferSize)
	{
	DWORD dwNumBytes;
	if (::WriteFile(hFile, (LPVOID)pbBuffer, bufferSize - compressionState.avail_out, &dwNumBytes, NULL))
		{
		compressionState.next_out = pbBuffer;
		compressionState.avail_out = dwNumBytes;
		}
	else
		throw ErrCannotWriteFile; 
	}

// ===========================================================================
// CSISFileGeneratorT<>
// Template class, handles the character size specific SIS file generation
// ===========================================================================

template<class T>
void CSISFileGeneratorT<T>::WriteString(LPCWSTR pStr, HANDLE hFile)
// Purpose  : Write a string to the file
// Inputs   : pStr	-	The string to write (comes from CSISWritre so is always UNICODE!)
//			  hFile -	The target file	
	{
	// Copy the string into the output format...
	long len = wcslen(pStr);


	T* buf = new T[len + 1];
	if (!buf) throw ErrNotEnoughMemory;
	
	for(int i = 0; i < len; ++i) {
	  unsigned short c = pStr[i];
	  flips(c); // Re-order unicode chars
	  buf[i] = (T)c;
	}
	// And write it to the file
	DWORD dwNumBytes;
	if(!::WriteFile(hFile, (LPVOID)buf, len * sizeof(T), &dwNumBytes, NULL))
		throw ErrFailedToWriteStringsBlock;
	
	// CRC
	Crc((void *)buf, len * sizeof(T));
	
	delete [] buf;
	}

template<class T>
void CSISFileGeneratorT<T>::WriteString8(LPCWSTR pStr, HANDLE hFile)
// Purpose  : Write an 8bit string to the file
// Inputs   : pStr	-	The string to write
//			  hFile -	The target file	
	{
	// Copy the string into the output format...
	long len = wcslen(pStr);
	
	char* buf = new char[len + 1];
	if (!buf) throw ErrNotEnoughMemory;
	
	for(int i = 0; i < len; ++i)
		buf[i] = (char)pStr[i];
	
	// And write it to the file
	DWORD dwNumBytes;
	if(!::WriteFile(hFile, (LPVOID)buf, len, &dwNumBytes, NULL))
		throw ErrFailedToWriteStringsBlock;
	
	// CRC
	Crc((void *)buf, len);
	
	delete [] buf;
	}


void CSISFileGeneratorBase::ParseCertificates()
	{
	// Test for PKCS7 chain and re-write as concatenated certificates
#ifdef DIGITAL_SIGNATURES
	//get digital sig info
	const SIGNATURENODE* pSignature = m_pSISWriter->GetSignatureBase();
	
	if (pSignature)
		{
		
		int status;
		
		m_bWriteChain = FALSE;
		
		//get a temporary filename for re-writing a PCKS chain
		LPWSTR pszTempCert;
		pszTempCert = TempFileName(pSignature->pszPublicKey);
		
		// convert tempfile name for Convert Signed method
		DWORD lent=wcslen(pszTempCert);
		LPSTR fNameTmp=MakeMBCSString(pszTempCert, CP_ACP, lent);
		
		
		// convert temporary Certificate file name for Convert Signed method
		DWORD lench=wcslen(pSignature->pszChainFile);
		LPSTR fNameChain=MakeMBCSString(pSignature->pszChainFile, CP_ACP, lench);
		
		
		// convert certificate name for Convert Signed method
		DWORD lenc=wcslen(pSignature->pszPublicKey);
		LPSTR fNameCrt=MakeMBCSString(pSignature->pszPublicKey, CP_ACP, lenc);
		
		
		// convert b64 name
		DWORD lencb64=wcslen(pSignature->pszB64File);
		LPSTR fNameB64=MakeMBCSString(pSignature->pszB64File, CP_ACP, lencb64);
		
		
		
		//create our PKCS7 aware object
		CPKCS7* pPKCS7 = new CPKCS7;
		if (!pPKCS7) throw ErrNotEnoughMemory;
		
		status = pPKCS7->ConvertSigned(fNameCrt, fNameTmp);
		//status = pPKCS7->ConvertBase64(fNameCrt, fNameTmp);
		
		
		// If not a single cert, specify that the end entity cert comes first.
		// Identify ee cert by private key
		CPrivateKey* pKey = LoadPrivateKey();
		pPKCS7->SetEndEntityKey(pKey);
		
		int keyMatchesCert = 1;
		if (status == PKCS7FileReWritten)
			{
			int extractStatus = pPKCS7->ExtractChain(fNameTmp, fNameChain, &(unsigned short)pSignature->iNumCerts);
			m_bWriteChain = TRUE;
			if (extractStatus == ErrEndEntityNotFound)
				keyMatchesCert = 0;
			
			}
		else if (status != PKCS7FileIsCertificate)
			{
			int extractStatus = pPKCS7->ExtractChain(fNameCrt, fNameChain, &(unsigned short)pSignature->iNumCerts);
			m_bWriteChain = TRUE;
			if (extractStatus == ErrEndEntityNotFound)
				keyMatchesCert = 0;
			}
		else
			{
			// File is a single cert - check it matches the key
			keyMatchesCert = pKey->MatchesCert(fNameCrt);
			}

		if (!keyMatchesCert)
			throw ErrCertFileKeyFileMismatch;
		
		delete pKey;
		delete pPKCS7;
		
		//create a Base64 convertor object
		CB64Decrypt* pB64 = new CB64Decrypt;
		
		if (!pB64) throw ErrNotEnoughMemory;
		
		if (m_bWriteChain)
			{
			pB64->Do(fNameChain, fNameB64, NULL);
			}
		else
			{
			pB64->Do(fNameCrt, fNameB64, NULL);
			}
		
		delete pB64;
		
		delete [] fNameTmp;
		delete [] fNameChain;
		delete [] fNameCrt;
		
		
		}
#endif
	}
