// CRTSIS.CPP
// Copyright (c) 1997-1999 Symbian Ltd. All rights reserved.
//
// Handles creation of a SIS file
//

// ===========================================================================
// INCLUDES
// ===========================================================================

#include "utils.h"

#include <windows.h>
#include <assert.h>

#include <tchar.h>

#include <wchar.h>
#include <time.h>
#include <stdio.h>
#include "crtsis.h"
#include "parsepkg.h"
#include "instendian.h"
#include "sisfilegen.h"



// ===========================================================================
// CSISWriter
// Holds SIS file info, as generated by the parser and builds the output SIS file.
// All strings are held internally by this class as UNICODE (wide) strings.
// ===========================================================================

CSISWriter::CSISWriter()
:m_dwUID(0x00000000),
m_wMajor(0),
m_wMinor(0),
m_dwBuild(0),
m_wType(EInstApplication),
m_wFlags(0),
m_wNumLangs(0),
m_pLangStringBase(NULL),
m_pLocalizedVendorBase(NULL),
m_pVendorName(NULL),
m_pLogoBase(NULL),
m_pLangBase(NULL),
m_pPkgLineBase(NULL),
m_pDependBase(NULL),
m_pPlatformDependBase(NULL),
m_pSignatureBase(NULL),
m_pCapabilityBase(NULL),
m_bCmdPassword(FALSE),
m_bSetTimestamp(FALSE),
m_bSelfsign(FALSE),
m_pObserver(NULL)

	{ ; }

void CSISWriter::Release()
// Dispose of components resources
	{
	// remove any temporary files
	CleanupTemp();
	
	// free up allocated structures
	LANGSTRINGNODE *pLangStringNode = m_pLangStringBase;
	while(m_pLangStringBase)
		{
		pLangStringNode = pLangStringNode->pNext;
		delete m_pLangStringBase;
		m_pLangStringBase = pLangStringNode;
		}
	
	LANGSTRINGNODE *pLocalizedVendorNode = m_pLocalizedVendorBase;
	while(m_pLocalizedVendorBase)
		{
		pLocalizedVendorNode = pLocalizedVendorNode->pNext;
		delete m_pLocalizedVendorBase;
		m_pLocalizedVendorBase = pLocalizedVendorNode;
		}
	
	LANGSTRINGNODE *pVendorNode = m_pVendorName;
	while(m_pVendorName)
		{
		pVendorNode = pVendorNode->pNext;
		delete m_pVendorName;
		m_pVendorName = pVendorNode;
		}

	LOGO *pLogoNode = m_pLogoBase;
	while(m_pLogoBase)
		{
		pLogoNode = pLogoNode->pNext;
		delete m_pLogoBase;
		m_pLogoBase = pLogoNode;
		}
	
	LANGNODE *pLangNode = m_pLangBase;
	while(m_pLangBase)
		{
		pLangNode = pLangNode->pNext;
		delete m_pLangBase;
		m_pLangBase = pLangNode;
		}
	
	PKGLINENODE *pPkgLineNode = m_pPkgLineBase;
	while(m_pPkgLineBase)
		{
		pPkgLineNode = m_pPkgLineBase->pNext;
		switch (m_pPkgLineBase->iPackageLineType)
			{
			case EInstPkgLineFile:
			case EInstPkgLineLanguageFile:
				delete m_pPkgLineBase->file;
				break;
			case EInstPkgLineOption:
				LANGSTRINGNODE *pLangStringNode2;
				pLangStringNode = m_pPkgLineBase->option->pLangStringBase;
				while(pLangStringNode)
					{
					pLangStringNode2 = pLangStringNode->pNext;
					delete pLangStringNode;
					pLangStringNode = pLangStringNode2;
					}
				delete m_pPkgLineBase->option;
				break;
			case EInstPkgLineCondIf:
			case EInstPkgLineCondElseIf:
				ReleaseCondExpr(m_pPkgLineBase->cond);
				break;
			case EInstPkgLineCondElse:
			case EInstPkgLineCondEndIf:
				break;
			}
		delete m_pPkgLineBase;
		m_pPkgLineBase = pPkgLineNode;
		}
	
	DEPENDNODE *pDependNode = m_pDependBase;
	while(m_pDependBase)
		{
		pDependNode = pDependNode->pNext;
		
		pLangStringNode = m_pDependBase->pLangStringBase;
		while(m_pDependBase->pLangStringBase)
			{
			pLangStringNode = pLangStringNode->pNext;
			delete m_pDependBase->pLangStringBase;
			m_pDependBase->pLangStringBase = pLangStringNode;
			}
		delete m_pDependBase;
		m_pDependBase = pDependNode;
		}
	
	DEPENDNODE *pPlatformDependNode = m_pPlatformDependBase;
	while(m_pPlatformDependBase)
		{
		pPlatformDependNode = pPlatformDependNode->pNext;
		
		pLangStringNode = m_pPlatformDependBase->pLangStringBase;
		while(m_pPlatformDependBase->pLangStringBase)
			{
			pLangStringNode = pLangStringNode->pNext;
			delete m_pPlatformDependBase->pLangStringBase;
			m_pPlatformDependBase->pLangStringBase = pLangStringNode;
			}
		delete m_pPlatformDependBase;
		m_pPlatformDependBase = pPlatformDependNode;
		}

	if (m_pSignatureBase) delete m_pSignatureBase;
	
	CAPABILITYNODE *pCapabilityNode = m_pCapabilityBase;
	while(m_pCapabilityBase)
		{
		pCapabilityNode = pCapabilityNode->pNext;
		delete m_pCapabilityBase;
		m_pCapabilityBase = pCapabilityNode;
		}
	
	}

void CSISWriter::ReleaseCondExpr(PKGLINECONDITION* expr)
// recursively called to free up condition expression
	{
	if (expr)
		{
		switch (expr->exprType)
			{
			case EInstCondBinOpEq:
			case EInstCondBinOpNe:
			case EInstCondBinOpGt:
			case EInstCondBinOpLt:
			case EInstCondBinOpGe:
			case EInstCondBinOpLe:
			case EInstCondLogOpAnd:
			case EInstCondLogOpOr:
				ReleaseCondExpr(expr->b.pLhs);
				ReleaseCondExpr(expr->b.pRhs);
				break;
			case EInstCondUnaryOpNot:
				ReleaseCondExpr(expr->pExpr);
				break;
			case EInstCondFuncAppProperties:
				ReleaseCondExpr(expr->pArg[0]);
				ReleaseCondExpr(expr->pArg[1]);
				break;
			case EInstCondFuncExists:
			case EInstCondFuncDevProperties:
				ReleaseCondExpr(expr->pArg[0]);
				break;
			case EInstCondPrimTypeString:
			case EInstCondPrimTypeVariable:
			case EInstCondPrimTypeNumber:
			case EInstCondPrimTypeOption:
				delete expr->pPrim;
				break;
			default:
				break;
			}
		delete expr;
		}
	}

void CSISWriter::SetCmdPassword(LPCWSTR pszCmdPassword)
	{
	wcscpy(m_pszCmdPassword, pszCmdPassword);
	m_bCmdPassword = TRUE;
	}

BOOL CSISWriter::GetCmdPassword(LPWSTR pszCmdPassword) const
	{
	wcscpy(pszCmdPassword, m_pszCmdPassword);
	
	return m_bCmdPassword;
	}

void CSISWriter::WriteSIS(LPCWSTR pszTargetFile, BOOL fIsStub /*=FALSE*/)
// Construct the output SIS file
// Inputs   : hFile	   The file handle to write to
//			  fIsStub  IS this a stub SIS file (stub SIS files have no CODE block)
	{
	CSISFileGenerator gen(this);
	gen.GenerateSISFile(pszTargetFile, fIsStub);
	}

void CSISWriter::AddLanguageNode(LANGNODE *pLNode)
// Adds a language node
// Inputs   : pLNode   The node to add
	{
	pLNode->pNext = m_pLangBase;
	m_pLangBase = pLNode;
	m_wNumLangs++;
	}

void CSISWriter::AddLangStringNode(LANGSTRINGNODE *pLSNode)
// Adds a language string node
// Inputs   : pLSNode   The node to add
	{
	pLSNode->pNext = m_pLangStringBase;
	m_pLangStringBase = pLSNode;
	}

void CSISWriter::AddLocalizedVendorNode(LANGSTRINGNODE *pLSNode)
// Adds a localized vendor name node
// Inputs   : pLSNode   The node to add
	{
	pLSNode->pNext = m_pLocalizedVendorBase;
	m_pLocalizedVendorBase = pLSNode;
	}

void CSISWriter::AddVendorNode(LANGSTRINGNODE *pLSNode)
// Adds a vendor name node
// Inputs   : pLSNode   The node to add
	{
	pLSNode->pNext = m_pVendorName;
	m_pVendorName = pLSNode;
	}

void CSISWriter::AddLogo(LOGO *pLSLogo)
// Sets a logo
// Inputs   : pLSLogo   The logo to add
	{
	pLSLogo->pNext = m_pLogoBase;
	m_pLogoBase = pLSLogo;
	}

void CSISWriter::AddPkgLineNode(PKGLINENODE *pNode)
// Adds a PKG line node
// Inputs   : pNode   The node to add
	{
	pNode->pNext = m_pPkgLineBase;
	m_pPkgLineBase = pNode;
	}

void CSISWriter::AddDependencyNode(DEPENDNODE *pNode)
// Does exactly what it says on the tin!
// Inputs   : pNode   The node to add
	{
	pNode->pNext = m_pDependBase;
	m_pDependBase = pNode;
	}

void CSISWriter::AddPlatformDependencyNode(DEPENDNODE *pNode)
// Does exactly what it says on the tin!
// Inputs   : pNode   The node to add
	{
	pNode->pNext = m_pPlatformDependBase;
	m_pPlatformDependBase = pNode;
	}

void CSISWriter::AddSignatureNode(SIGNATURENODE *pSig)
// Sets the digital signature
// Inputs   : pSig   The node to add
	{
	m_pSignatureBase = pSig;
	}

void CSISWriter::AddCapabilityNode(CAPABILITYNODE *pNode)
// Does exactly what it says on the tin!
// Inputs   : pNode   The node to add
	{
	pNode->pNext = m_pCapabilityBase;
	m_pCapabilityBase = pNode;
	}

BOOL CSISWriter::AreLangStringInit()
// Is there a language string node defined? (i.e. has the pkg language header line been parsed?)
	{
	return (m_pLangStringBase!=NULL);
	}

BOOL CSISWriter::AreLocalizedVendorNamesInit()
// Is there a localized vendor name defined?
	{
	return (m_pLocalizedVendorBase!=NULL);
	}

BOOL CSISWriter::IsVendorNameInit()
// Is there a vendor name defined?
	{
	return (m_pVendorName!=NULL);
	}

void CSISWriter::SetVersionInfo(DWORD dwUID, WORD wMajor, WORD wMinor, DWORD dwBuild, TSISType wType, WORD wFlags)
// Inputs   : dwUID							Installee app's uid
//			  dwMajor, dwMinor, dwBuild		The version numbers
//		      wType							SIS file type
//		      wFlags						Installation flags
	{
	m_dwUID		= dwUID;    
	m_wMajor	= wMajor;  
	m_wMinor	= wMinor;  
	m_dwBuild	= dwBuild;
	m_wType		= wType;
	m_wFlags	= wFlags;  
	}

void CSISWriter::QueryVersionInfo (WORD *wMajor, WORD *wMinor, DWORD *dwBuild) const
// Inputs   : dwMajor, dwMinor, dwBuild	The destinations for the version numbers
	{
	*wMajor   = m_wMajor;
	*wMinor   = m_wMinor;
	*dwBuild   = m_dwBuild;
	}

void CSISWriter::SetLangDependFileDestinations(LPCWSTR pszDest, LPCWSTR pszMimeType, DWORD dwType, DWORD dwOptions)
// Purpose  : Patches the last 'numlang' files with a target path and flags (including the
//			  OPTFILE_LANGUAGEDEPENDENT flag)
// Inputs   : pszDest	The target path
//	  : pszMimeType The mime type (if relevant)
//			  dwOptions Any specified flags
	{
	PKGLINENODE *pNode = m_pPkgLineBase;
	WORD wNumLangs = 0;
	
	while(pNode && (wNumLangs++ < m_wNumLangs))
		{
		wcscpy(pNode->file->pszDest, pszDest);
		pNode->iPackageLineType = EInstPkgLineLanguageFile;
		pNode->file->type = (TInstFileType)dwType;
		switch (pNode->file->type)
			{
			case EInstFileTypeText:
				pNode->file->options.iTextOption=(TInstFileTextOption)dwOptions;
				break;
			case EInstFileTypeRun:
				pNode->file->options.iRunOption=(TInstFileRunOption)dwOptions;
				break;
			case EInstFileTypeMime:
				pNode->file->options.iMimeOption=(TInstFileMimeOption)dwOptions;
				wcscpy(pNode->file->pszMimeType, pszMimeType);
				break;
			default:
				break;
			}
		pNode = pNode->pNext;
		}
	}

void CSISWriter::SetLangDependCompDestinations(DWORD dwUID)
// Purpose  : Sets the component id of the last 'numlang' file nodes, and sets their flags to
//			  OPTFILE_LANGUAGEDEPENDENT | OPTFILE_COMPONENT;
// Inputs   : dwUID  The components UID
	{
	PKGLINENODE *pNode = m_pPkgLineBase;
	WORD wNumLangs = 0;
	
	while(pNode && (wNumLangs++ < m_wNumLangs))
		{
		pNode->iPackageLineType = EInstPkgLineLanguageFile;
		pNode->file->type = EInstFileTypeComponent;
		pNode->file->options.iComponentUid = dwUID;
		pNode = pNode->pNext;
		}
	}

void CSISWriter::SetTimestamp(DWORD dwYear, DWORD dwMonth, DWORD dwDay, DWORD dwHours, DWORD dwMinutes, DWORD dwSeconds) {
	m_bSetTimestamp = TRUE;
	m_dwYear = dwYear;
	m_dwMonth = dwMonth;
	m_dwDay = dwDay;
	m_dwHours = dwHours;
	m_dwMinutes = dwMinutes;
	m_dwSeconds = dwSeconds;
}

BOOL CSISWriter::GetTimestamp(DWORD* dwYear, DWORD* dwMonth, DWORD* dwDay, DWORD* dwHours, DWORD* dwMinutes, DWORD* dwSeconds) const {
	if (!m_bSetTimestamp)
		return FALSE;
	*dwYear = m_dwYear;
	*dwMonth = m_dwMonth;
	*dwDay = m_dwDay;
	*dwHours = m_dwHours;
	*dwMinutes = m_dwMinutes;
	*dwSeconds = m_dwSeconds;
	return TRUE;
}

void CSISWriter::SetSelfsign(BOOL selfsign) {
	m_bSelfsign = selfsign;
}

